import logging
import os
import re
from typing import Iterable

from openpyxl.workbook.child import INVALID_TITLE_REGEX

from django.core import files
from django.core.files.temp import NamedTemporaryFile
from django.db import transaction
from django.utils import timezone

from kelvin.common.yql import YqlOperationResultsRequest, YqlOperationStatusRequest
from kelvin.common.yql import client as yql_client
from kelvin.resources.models import Resource

from . import signals
from .mails import CourseReportRequestEmail
from .models import CourseReportRequest, Operation, Query, ResultData

logger = logging.getLogger(__name__)


@transaction.atomic()
def create_operation_by_course_report_request(course_report_req_id: int) -> None:
    report_request: CourseReportRequest = CourseReportRequest.objects\
        .select_for_update()\
        .select_related('query')\
        .get(pk=course_report_req_id)

    if report_request.operation_id:
        return

    query = report_request.query
    operation = Operation.objects.create(
        query=query,
        action=Operation.ACTION_RUN,
        query_data={
            'type': query.type,
            'content': query.content,
            'parameters': report_request.parameters,
        },
        query_title=query.title,
        query_version=query.version if query.version else None,
        created_by=report_request.user,
    )
    report_request.operation = operation
    report_request.save()


def update_course_report_requests_by_operation(operation_id: int, status: str) -> None:
    CourseReportRequest.objects.filter(
        operation_id=operation_id,
    ).update(
        status=status
    )


def validate_query(query_id: int) -> Operation:
    query = Query.objects.get(pk=query_id)
    return Operation.objects.create(
        query=query,
        action=Operation.ACTION_VALIDATE,
        query_data={
            'type': query.type,
            'content': query.content,
            'parameters': {},
        },
        query_title=query.title,
        created_by=query.created_by,
    )


@transaction.atomic()
def create_yql_operation(operation_id: int) -> None:
    op: Operation = Operation.objects.select_for_update().select_related('query').get(
        pk=operation_id,
        status=Operation.STATUS_IDLE,
    )

    operation_data = op.operation_data
    query = operation_data['content']
    parameters = operation_data['parameters'] if operation_data['parameters'] else None

    request = yql_client.query(query=query, syntax_version=1)
    request.run(parameters=parameters)

    op.yql_operation_id = request.operation_id
    op.status = request.status
    op.save()


@transaction.atomic()
def check_yql_operation(operation_id: int) -> None:
    op = Operation.objects.select_for_update(skip_locked=True).get(
        pk=operation_id,
    )

    if not op.yql_operation_id:
        logger.warning('Operation {} has no yql_operation_id'.format(operation_id))
        return

    request = YqlOperationStatusRequest(req_id=op.yql_operation_id)
    request.run()

    op.status = request.status

    if op.status in Operation.FINAL_STATUSES:
        op.errors = [str(e) for e in request.errors]
        op.issues = [str(e) for e in request.issues]

    op.save()


def check_yql_operation_chunk(operations: Iterable):
    for operation_id in operations:
        try:
            check_yql_operation(operation_id)
        except Exception as exc:
            logger.exception(exc)
            pass


def save_table_as_file(table, output_type='xlsx', colnames=None, **kwargs):
    from pyexcel import get_sheet, free_resources

    dest_file = NamedTemporaryFile(delete=True)
    sheet = get_sheet(
        array=table.rows,
    )

    if colnames:
        new_colnames = [colnames.get(c, c) for c in table.column_names]
        sheet.colnames = new_colnames
    else:
        sheet.colnames = table.column_names

    sheet_name = kwargs.pop('sheet_name', None)
    if sheet_name:
        sheet.name = re.sub(INVALID_TITLE_REGEX, '_', sheet_name)

    sheet.save_as(
        dest_file.name,
        force_file_type=output_type
    )

    free_resources()

    return dest_file


@transaction.atomic()
def download_yql_result_data(operation_id: int) -> None:
    op = Operation.objects.get(pk=operation_id)

    if op.results.exists() or not op.status == Operation.STATUS_COMPLETED:
        return

    report_request: CourseReportRequest = op.course_report_requests.select_for_update(skip_locked=True).first()
    if not report_request:
        return

    request = YqlOperationResultsRequest(req_id=op.yql_operation_id)
    request.run()

    results = list(request.get_results())
    if not len(results):
        return

    table = results[0]
    table.fetch_full_data()

    filename = '{}_{}.{}'.format(
        op.query_title,
        str(int(report_request.created.timestamp())),
        report_request.format.lower(),
    )

    colnames = dict(op.query.output_fields.values_list('name', 'title'))

    extra = dict()

    if report_request.format == CourseReportRequest.FORMAT_XLSX:
        extra['sheet_name'] = op.query_title

    temp_file = save_table_as_file(
        table,
        output_type=report_request.format,
        colnames=colnames if colnames else None,
        **extra,
    )

    temp_file.flush()
    filesize = os.stat(temp_file.name).st_size

    resource = Resource.objects.create(
        file=files.File(temp_file, name=filename),
    )
    result_data = ResultData.objects.create(
        operation=op,
        format=report_request.format,
        resource=resource,
        size=filesize,
    )

    report_request.result_data = result_data
    report_request.save()

    signals.download_result_data_completed.send(
        sender=report_request.__class__,
        report=report_request,
    )


def send_report_notification(report_id: int) -> None:
    report = CourseReportRequest.objects.get(pk=report_id)

    if not report.notify_by_email or report.notified:
        return

    CourseReportRequestEmail(report).send()
    CourseReportRequest.objects.filter(pk=report_id).update(notified=timezone.now())
