import logging
from datetime import timedelta
from enum import Enum
from typing import Iterable

from django.utils import timezone

from kelvin.celery import app
from kelvin.common.decorators.task_logger import logged_task

from . import services
from .models import Operation
from .reports import ClickHouseIterator, CsvReporter, FilesReporter, SqlIterator
from .utils import send_report

logger = logging.getLogger(__name__)


class OutputFormat(Enum):
    CSV = 'csv'


@app.task()
def sql_report(name, query, receivers, output_format=OutputFormat.CSV):
    """
    Celery-task для генерации SQL отчета
    :param name: str Название отчета
    :param query: str SQL-запрос
    :param receivers: list[str] Список email для уведомления
    :param output_format: OutputFormat Формат для сохранения результата
    """
    output_format = OutputFormat(output_format)

    if output_format == OutputFormat.CSV:
        reporter = CsvReporter()
    else:
        raise ValueError(
            'Unsupported output format "{}"'.format(output_format)
        )

    sql_result = SqlIterator(query)
    files = reporter.from_iterable(name, sql_result)

    send_report(receivers, name, files)


@app.task()
def clickhouse_report(name, query, filename, receivers, timeout=None):
    """
    Celery-task для генерации ClickHouse отчета
    :param name: str Название отчета
    :param query: str ClickHouse запрос
    :param filename: str Имя файла для сохранения результата
    :param receivers: list[str] Список email для уведомления
    :param timeout: int Timeout для запроса в ClickHouse
    """
    ch_result = ClickHouseIterator([(filename, query)], timeout=timeout)

    reporter = FilesReporter()
    files = reporter.from_iterable(name, ch_result)

    send_report(receivers, name, files)


@logged_task
@app.task()
def create_yql_operation(operation_id: int):
    services.create_yql_operation(operation_id)


@logged_task
@app.task()
def check_yql_operation(operation_id: int):
    services.check_yql_operation(operation_id)


@logged_task
@app.task(soft_time_limit=60 * 30, time_limit=60 * 40)
def download_yql_result_data(operation_id: int):
    services.download_yql_result_data(operation_id)


@logged_task
@app.task()
def check_yql_operation_chunk(operations: Iterable):
    services.check_yql_operation_chunk(operations)


@logged_task
@app.task()
def sync_yql_operations(chunk_size=10, lag=60, limit=1000):
    lag_time = timezone.now() - timedelta(seconds=lag)
    queryset = Operation.objects.filter(
        yql_operation_id__isnull=False,
        modified__lt=lag_time,
    ).exclude(
        status__in=Operation.FINAL_STATUSES,
    ).values_list(
        'id', flat=True
    ).order_by('modified')

    operation_ids = list(queryset[:limit])
    logger.info('{} incompleted operation found'.format(len(operation_ids)))

    for i in range(0, len(operation_ids), chunk_size):
        chunk = operation_ids[i:i + chunk_size]
        logger.info('chunk {} - [{}]'.format(i, str(chunk)))
        check_yql_operation_chunk.delay(operations=chunk)


@logged_task
@app.task()
def send_report_notification(report_id: int):
    services.send_report_notification(report_id)
