from rest_condition import Or

from rest_framework import permissions, status
from rest_framework.generics import ListAPIView, ListCreateAPIView, RetrieveAPIView, get_object_or_404
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response
from rest_framework.views import APIView

from kelvin.accounts.permissions import (
    IsContentManager, IsCurator, IsStaff, IsTeacher, ObjectForContentManager, ObjectForOwner, ObjectForStaff,
    ObjectForTeacher,
)
from kelvin.courses.models import Course

from .models import CourseReportRequest, Query, ResultFormatMixin
from .serializers import (
    CourseReportRequestCreateSerializer, CourseReportRequestSerializer, CourseReportResultsDataSerializer,
    QuerySerializer,
)


class QueryListAPIView(ListAPIView):
    permission_classes = (IsAuthenticated,)
    queryset = Query.objects.active()
    serializer_class = QuerySerializer
    pagination_class = None


class QueryDetailAPIView(RetrieveAPIView):
    permission_classes = (IsAuthenticated,)
    queryset = Query.objects.active()
    serializer_class = QuerySerializer


class ReportFormatAPIView(APIView):
    permission_classes = (IsAuthenticated,)
    pagination_class = None

    def get(self, *args, **kwargs):
        formats = [k for k, v in ResultFormatMixin.get_format_choices()]
        return Response(formats)


class CourseReportRequestView(ListCreateAPIView):
    queryset = CourseReportRequest.objects.prefetch_related('operation').visible()
    serializer_class = CourseReportRequestSerializer
    course_url_kwarg = 'course_id'
    permission_classes = [
        IsAuthenticated,
        Or(IsTeacher, IsContentManager, IsCurator, IsStaff),
        Or(ObjectForTeacher, ObjectForContentManager, ObjectForStaff, ObjectForOwner),
    ]

    def get_course_id(self):
        return self.kwargs.get(self.course_url_kwarg)

    def check_object_permissions(self, request, obj):
        course = obj.course
        return super().check_object_permissions(request, course)

    def get_queryset(self):
        queryset = super().get_queryset()

        course_id = self.get_course_id()
        course = get_object_or_404(Course, pk=course_id)
        queryset = queryset.filter(course=course)

        return queryset

    def create(self, request, *args, **kwargs):
        input_serializer = CourseReportRequestCreateSerializer(data=request.data)
        input_serializer.is_valid(raise_exception=True)
        instance = input_serializer.save(user=request.user, course_id=self.get_course_id())
        serializer = self.get_serializer(instance)
        return Response(serializer.data, status=status.HTTP_201_CREATED)


class CourseReportRequestDetailView(RetrieveAPIView):
    queryset = CourseReportRequest.objects.visible()
    serializer_class = CourseReportRequestSerializer
    permission_classes = [
        IsAuthenticated,
        Or(IsTeacher, IsContentManager, IsCurator, IsStaff),
        Or(ObjectForTeacher, ObjectForContentManager, ObjectForStaff, ObjectForOwner),
    ]


class CourseReportResultDataDetailView(RetrieveAPIView):
    queryset = CourseReportRequest.objects.prefetch_related('result_data').visible()
    serializer_class = CourseReportResultsDataSerializer
    permission_classes = [
        IsAuthenticated,
        Or(IsTeacher, IsContentManager, IsCurator, IsStaff),
        Or(ObjectForTeacher, ObjectForContentManager, ObjectForStaff, ObjectForOwner),
    ]

    def retrieve(self, request, *args, **kwargs):
        instance: CourseReportRequest = self.get_object()
        serializer = self.get_serializer(instance.result_data)
        return Response(serializer.data)
