import re
from builtins import object, str

from dal import autocomplete

from django import forms
from django.conf.urls import url
from django.contrib import admin
from django.core.urlresolvers import reverse
from django.db.models.aggregates import Count
from django.forms import fields
from django.utils.translation import ugettext_lazy as _

from kelvin.common.admin import AvailableForSupportAdminMixin, UserBlameModelAdminMixin
from kelvin.common.s3upload.widgets import S3UploadWidget
from kelvin.resources.models import Resource, ResourceTag, ScormResource
from kelvin.switcher import get_feature_bool_option


class ResourceTagAutocomplete(autocomplete.Select2QuerySetView):
    """Вью для получения списка тегов в формах с автокомплитом"""

    def get_queryset(self):
        """
        Возвращает данные только пользователям с доступом в админку.
        Фильтрует теги по имени
        """
        if not self.request.user.is_staff:
            return ResourceTag.objects.none()

        if self.q:
            return ResourceTag.objects.filter(name__icontains=self.q)
        else:
            return ResourceTag.objects.none()


def file_preview_url(url):
    """ Возвращает превью картинки или просто текстовую ссылку на объект """
    url_template = u'<a href={0} target="_blank">{1}</a>'
    if url.endswith(('.png', '.jpg', '.jpeg', '.svg', '.gif')) \
            or 'switchman' in url:
        img = u'<img style="display: block; width: auto; height: auto;' \
              u'max-width: 100%; max-height:100px;" src="{0}"/>'.format(url)
        return url_template.format(url, img)
    return url_template.format(url, url.split('/')[-1])


class PreviewClearableFileInput(fields.ClearableFileInput):
    """ Поле загрузки файла, которое умеет показывать превью изображений """

    def render(self, name, value, attrs=None):
        result = super(PreviewClearableFileInput, self).render(
            name, value, attrs)
        if value:
            result = u'{0}<br />{1}'.format(
                result, file_preview_url(value.url))
        return result


class ProblemAdminForm(forms.ModelForm):
    """
    Форма для админки задачи
    """
    file = fields.FileField(widget=PreviewClearableFileInput)

    class Meta(object):
        model = Resource
        fields = '__all__'


class ResourceAdminForm(forms.ModelForm):
    """
    Форма для админки ресурса
    """
    file = fields.FileField(widget=PreviewClearableFileInput)

    class Meta:
        model = Resource
        fields = '__all__'


class ScormResourceAdminForm(forms.ModelForm):
    """
    Форма для админки ресурса
    """
    s3_file = forms.URLField(widget=S3UploadWidget(dest='resources'))

    class Meta:
        model = ScormResource
        exclude = ('file',)


class ImageSizeFilter(admin.SimpleListFilter):
    """
    Фильтр по размерам картинок
    """
    title = _(u'Размеры картинок')
    parameter_name = 'dimensions'
    LOOKUP_CHOICES = (
        ('100', _(u'До 100x100')),
        ('300', _(u'100x100 - 300x300')),
        ('1000', _(u'300x300 - 1000x1000')),
        ('big', _(u'1000x1000 и больше')),
    )
    BOUNDS_DICT = {
        '100': (None, 100),
        '300': (100, 300),
        '1000': (300, 1000),
        'big': (1000, None),
    }

    def lookups(self, request, model_admin):
        """
        Возвращает список параметров запроса и их представлений
        """
        return self.LOOKUP_CHOICES

    def queryset(self, request, queryset):
        """
        Фильтрует картинки по размерам из запроса
        При этом картинка с размерами 80x1200 будет как в категории "меньше
        100", так и в "больше 1000"
        """
        bounds = self.BOUNDS_DICT.get(self.value())
        if not bounds:
            return queryset
        filter_width = queryset
        filter_height = queryset
        minimum, maximum = bounds
        if minimum:
            filter_width = filter_width.filter(image_width__gte=minimum)
            filter_height = filter_height.filter(image_height__gte=minimum)
        if maximum:
            filter_width = filter_width.filter(image_width__lte=maximum)
            filter_height = filter_height.filter(image_height__lte=maximum)
        return filter_width | filter_height


class ResourceAdmin(UserBlameModelAdminMixin, AvailableForSupportAdminMixin, admin.ModelAdmin):
    """
    Админка для ресурсов задач
    """
    list_display = (
        'action_checkbox',
        'id',
        'name',
        'get_image_dimensions',
        'get_url',
        'nda',
        'get_file_url',
    )
    list_filter = (
        ImageSizeFilter,
        'nda',
    )
    filter_horizontal = ('tags',)
    readonly_fields = (
        'get_problem_links',
        'get_file_url',
        'created_by',
        'modified_by',
    )
    form = ResourceAdminForm
    ordering = ('-id',)
    list_display_links = (
        'id',
        'name',
    )

    def get_url(self, obj):
        """
        Возвращает ссылку на ресурс в виде картинки или текстовой ссылки
        """
        if obj.shortened_file_url and get_feature_bool_option('E7N_ENABLE_NEW_RESOURCE_URLS', False):
            return file_preview_url(obj.shortened_file_url)

        return file_preview_url(obj.get_content_url())

    get_url.short_description = u'Адрес ресурса'
    get_url.allow_tags = True

    def get_file_url(self, obj):
        if obj.shortened_file_url and get_feature_bool_option('E7N_ENABLE_NEW_RESOURCE_URLS', False):
            return obj.shortened_file_url

        return obj.get_content_url()

    get_file_url.short_description = u'Ссылка на ресурс'
    get_file_url.admin_order_field = 'file'

    def get_image_dimensions(self, obj):
        """
        Размеры изображения, если есть
        """
        if obj.image_width and obj.image_height:
            return u"{0}x{1}".format(obj.image_width, obj.image_height)
        return None

    get_image_dimensions.short_description = u'Размеры'
    get_image_dimensions.admin_order_field = 'image_width'

    def get_problem_links(self, obj):
        """
        Ссылки на ресурсы
        """
        return u'\n'.join([
            u'<p><a href={0} target="_blank">{1}</a></p>'
            .format(reverse('admin:problems_problem_change',
                            args=(problem.id,)), str(problem))
            for problem in obj.problem_set.all().only('id')
        ])
    get_problem_links.short_description = _(u'Задачи, использующие ресурс')
    get_problem_links.allow_tags = True

    def get_queryset(self, request):
        """
        Если запрос на получение queryset пришел от метода POST, то проверяем,
        содержатся ли id ресурсов в запросе. Если да - возвращаем только их.
        Иначе возвращаем весь queryset.
        """
        qs = super(ResourceAdmin, self).get_queryset(request)
        if request.method == 'POST':
            resource_ids = []
            # post params with ids looks like 'form-12-id': '12345'
            pattern = re.compile(r'form-\d+-id')
            for param in request.POST:
                if pattern.match(param):
                    resource_ids.append(request.POST[param])
            if resource_ids:
                return qs.filter(id__in=resource_ids)
        return qs.prefetch_related('tags')


@admin.register(ScormResource)
class ScormResourceAdmin(ResourceAdmin):
    list_display = (
        'id',
        'name',
        'get_file_url',
        'status',
    )
    fields = (
        'name',
        's3_file',
        'nda',
        'shortened_file_url',
        'status',
        'error_messages',
        'public_url',
        'get_file_url',
        'available_for_support',
        'created_by',
        'modified_by',
    )
    list_filter = []
    form = ScormResourceAdminForm


class SortByResourceCountFilter(admin.SimpleListFilter):
    """
    Фильтр-замена сортировки по чсилу ресурсов
    """
    title = _(u'Сортировка по количеству ресурсов')
    parameter_name = 'resources_count'
    LOOKUP_CHOICES = (
        ('0', u'По возрастанию'),
        ('1', u'По убыванию'),
    )

    def lookups(self, request, model_admin):
        """
        Возвращает список параметров запроса и их представлений
        """
        return self.LOOKUP_CHOICES

    def queryset(self, request, queryset):
        """
        Добавляет сортировку по количеству ресурсов
        """
        value = self.value()
        if value is None:
            return queryset
        elif value == '1':
            return queryset.order_by('-resource_count')
        else:
            return queryset.order_by('resource_count')


class ResourceTagAdmin(admin.ModelAdmin):
    """
    Админка для тегов ресурсов задач
    """
    list_display = (
        'id',
        'name',
        'get_dependent_tags',
        'get_resource_count',
    )
    search_fields = (
        'name',
    )
    list_filter = (
        SortByResourceCountFilter,
    )
    filter_horizontal = (
        'dependent_tags',
    )

    def get_queryset(self, request):
        return (super(ResourceTagAdmin, self).get_queryset(request)
                .prefetch_related('dependent_tags')
                .annotate(resource_count=Count('resource')))

    def get_resource_count(self, obj):
        """
        Показывает количество использований тэга, возвращает ссылку на
        фильтрацию ресурсов по этому тегу
        """
        return (
            u'<a href="/admin/resources/resource/?tags__id__exact={0}" '
            u'target="_blank">{1}</a>'
            .format(obj.id, obj.resource_count)
        )
    get_resource_count.short_description = u'Количество использований тэга'
    get_resource_count.allow_tags = True

    def get_dependent_tags(self, obj):
        """
        Список зависимых тегов
        """
        return u', '.join(obj.dependent_tags.values_list('name', flat=True))
    get_dependent_tags.short_description = _(u'Зависимые тэги')

    def get_urls(self):
        """
        Добавляет урл для автокомплита тегов
        """
        info = self.model._meta.app_label, self.model._meta.model_name
        autocomplete_url = url(
            r'autocomplete/$',
            self.admin_site.admin_view(ResourceTagAutocomplete.as_view()),
            name='{0}_{1}_autocomplete'.format(*info),
        )
        return [autocomplete_url] + super(ResourceTagAdmin, self).get_urls()


admin.site.register(ResourceTag, ResourceTagAdmin)
admin.site.register(Resource, ResourceAdmin)
