import csv
import logging
from collections import Counter

from tqdm import tqdm

from django.conf import settings
from django.core.management.base import BaseCommand

from kelvin.common.s3.s3_boto_client import s3_boto_client
from kelvin.resources.models import Resource

logger = logging.getLogger(__name__)


class Command(BaseCommand):
    help = "Delete unused and unregistered resources"

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.counter = Counter()

    def add_arguments(self, parser):
        parser.add_argument(
            'file',
            nargs='?',
            help='CSV file containing a list of unused resources'
        )

        parser.add_argument(
            '--search-unpacked',
            action='store_true',
            help='Search for unpacked files by resource id'
        )

        parser.add_argument(
            '--search-unregistered',
            action='store_true',
            help='Search for unregistered resources in DB'
        )

        parser.add_argument(
            '--dry-run',
            action='store_true',
            help='Show unused resources without deletion'
        )

    def process_file(self, file_path, size=None, dry_run=False, verbosity=1):
        try:
            if not size:
                size = s3_boto_client.file_size(file_path)

            if not dry_run:
                s3_boto_client.copy_object(file_path, f"{settings.S3_TRASH_PATH}/{file_path}")

            if verbosity > 1:
                self.stdout.write(f"{file_path}: {size}")

            self.counter['total_size'] += size
            self.counter['total_objects'] += 1

            return True

        except Exception as e:
            self.stderr.write(str(e))
            self.counter['errors'] += 1

        return False

    def delete_objects_from_list(self, list_objects):
        try:
            result = s3_boto_client.delete_objects(list_objects)
            if result.get('Errors'):
                self.stderr.write('Errors occurred during deletion:')
                for error in result['Errors']:
                    self.stderr.write(str(error))
                self.counter['errors'] += len(result['Errors'])
        except Exception as e:
            self.stderr.write(str(e))
            self.counter['errors'] += len(list_objects)

    def handle(self, *args, **options):
        if options['verbosity'] and options['dry_run']:
            self.stdout.write('Dry run without deletion')

        if options['file']:
            with open(options['file'], newline='') as csv_file:
                reader = csv.DictReader(csv_file, delimiter=',')
                if options['verbosity']:
                    self.stdout.write(f"Searching unused resources from {options['file']}...")
                    self.stdout.write(f"Trash path: {settings.S3_TRASH_PATH}")
                    if options['search_unpacked']:
                        self.stdout.write(f"Unpacked resources folder path: {settings.S3_SCORM_RESOURCE_PATH}")

                for resource in tqdm(reader) if options['verbosity'] else reader:
                    files = []

                    processed = self.process_file(
                        file_path=resource['file'],
                        dry_run=options['dry_run'],
                        verbosity=options['verbosity']
                    )

                    if not options['dry_run'] and processed:
                        files.append(resource['file'])

                    if options['search_unpacked']:
                        try:
                            resource_dir = f"{settings.S3_SCORM_RESOURCE_PATH}/{resource['id']}/"
                            unpacked_resources = s3_boto_client.list(resource_dir).get('Contents', [])
                        except Exception as e:
                            self.stderr.write(str(e))
                            self.counter['errors'] += 1
                        else:
                            for res in unpacked_resources:
                                processed = self.process_file(
                                    file_path=res['Key'],
                                    size=res['Size'],
                                    dry_run=options['dry_run'],
                                    verbosity=options['verbosity']
                                )

                                if not options['dry_run'] and processed:
                                    files.append(res['Key'])

                    if not options['dry_run']:
                        self.delete_objects_from_list(files)

        if options['search_unregistered']:
            if options['verbosity']:
                self.stdout.write("Searching for unregistered resources...")
                self.stdout.write(f"Resource folder: {settings.S3_SCORM_RESOURCE_PATH}")

            prefix = f"{settings.S3_SCORM_RESOURCE_PATH}/"
            prefix_len = len(prefix)

            s3_resources = s3_boto_client.list(prefix, '.').get('Contents', [])
            resource_ids = set()
            for s3_resource in s3_resources:
                try:
                    folder = s3_resource['Key'][prefix_len:].split('/')[0]
                    if folder:
                        resource_ids.add(int(folder))
                except Exception as e:
                    self.stderr.write(str(e))

            for resource_id in tqdm(resource_ids) if options['verbosity'] else resource_ids:
                resource = Resource.objects.filter(id=resource_id)
                if not resource:
                    resource_folder_path = f"{prefix}{resource_id}/"
                    resources_list = s3_boto_client.list(resource_folder_path).get('Contents', [])
                    files = []
                    for res in resources_list:
                        processed = self.process_file(
                            file_path=res['Key'],
                            size=res['Size'],
                            dry_run=options['dry_run'],
                            verbosity=options['verbosity']
                        )

                        if not options['dry_run'] and processed:
                            files.append(res['Key'])

                    if not options['dry_run']:
                        self.delete_objects_from_list(files)

        if options['verbosity']:
            self.stdout.write(f"Objects found: {self.counter['total_objects']}, "
                              f"total size: {self.counter['total_size']}.")
            self.stdout.write(f"Errors: {self.counter['errors']}.")
