import os
import uuid
from builtins import object, str
from urllib.parse import urlparse

from model_utils import FieldTracker

from django.conf import settings
from django.db import models
from django.utils.encoding import python_2_unicode_compatible
from django.utils.translation import ugettext_lazy as _

from kelvin.common.model_mixins import AvailableForSupportMixin, TimeStampMixin, UserBlameMixin
from kelvin.common.s3upload.fields import S3UploadField
from kelvin.common.utils import safe_filename


def resource_upload_path(instance, filename):
    """
    Возвращает путь и имя файла для сохранения ресурса
    """
    return safe_filename(filename, 'resources')


@python_2_unicode_compatible
class Resource(TimeStampMixin, AvailableForSupportMixin, UserBlameMixin, models.Model):
    """
    Переиспользуемые ресурсы
    """
    name = models.CharField(
        verbose_name=_('Название'),
        max_length=255,
        default='',
        blank=True
    )
    deprecated = models.BooleanField(
        verbose_name=_('Устарел'),
        default=False,
    )
    file = models.FileField(
        verbose_name=_('Файл ресурса'),
        null=True,
        blank=True,
        upload_to=resource_upload_path,
        db_index=True,
    )
    image_width = models.PositiveIntegerField(
        verbose_name=_('Ширина файла-картинки'),
        blank=True,
        null=True,
    )
    image_height = models.PositiveIntegerField(
        verbose_name=_('Высота файла-картинки'),
        blank=True,
        null=True,
    )
    tags = models.ManyToManyField(
        'ResourceTag',
        verbose_name=_('Тэги'),
        blank=True,
    )
    nda = models.BooleanField(
        verbose_name=_('Недоступен внешним слушателям'),
        default=False
    )
    shortened_file_url = models.CharField(
        verbose_name=_('Короткая ссылка на ресурс'),
        help_text=_('Сохраняется автоматически'),
        max_length=1024,
        default='',
        blank=True,
    )
    # Для отслеживания изменения файла-картинки
    tracker = FieldTracker(fields=[
        'file',
    ])

    class Meta(object):
        verbose_name = _('Ресурс')
        verbose_name_plural = _('Ресурсы')

    def __str__(self):
        """
        Строковое представление ресурса
        """
        return '{0}: {1}: {2}{3}'.format(
            self.id,
            self.name,
            self.get_content_url(),
            ' [NDA]' if self.nda else ''
        )

    def get_content_url(self):
        """
        Возвращает ссылку на файл, если есть файл, либо `url`.
        """
        # TODO: сделать поле file обязательным
        return self.file.url if self.file else ''

    def get_type(self):
        """
        Возвращает тип ресурса - расширение файла
        """
        if not self.file:
            return ''
        name, extension = os.path.splitext(self.file.name)
        # первый символ - точка
        return extension[1:] if extension else 'other'

    def save(self, *args, **kwargs):
        """
        Переопределяем стандартный метод Save,
        После вызова оригинального save добавляем
        логику генерации shortened-ссылки на ресурс
        """

        if not self.shortened_file_url:
            self.shortened_file_url = "{0}/{1}".format(
                settings.SWITCHMAN_URL,
                uuid.uuid4()
            )
        super(Resource, self).save(*args, **kwargs)


class ScormResource(Resource):
    SCORM_MODULE_STATUS_PENDING = 'pending'
    SCORM_MODULE_STATUS_READY = 'ready'
    SCORM_MODULE_STATUS_ERROR = 'error'

    SCORM_MODULE_STATUS_CHOICES = (
        (SCORM_MODULE_STATUS_PENDING, _("в обработке")),
        (SCORM_MODULE_STATUS_READY, _("готов")),
        (SCORM_MODULE_STATUS_ERROR, _("ошибка")),
    )
    s3_file = S3UploadField(
        dest='resources',
        verbose_name=_("Файл scorm-ресурса на s3"),
        blank=True,
        null=True,
    )
    status = models.CharField(
        _("статус"),
        max_length=10,
        choices=SCORM_MODULE_STATUS_CHOICES,
        default=SCORM_MODULE_STATUS_PENDING,
        help_text=_("статус распаковки архива с курсом и готовности к использованию"),
    )
    error_messages = models.TextField(
        _("ошибки распаковки архива и разбора манифеста"),
        blank=True,
    )

    public_url = models.CharField(
        _("распакованный курс"),
        max_length=255,
        blank=True,
        help_text=_('путь к папке с распакованным scorm-курсом'),
    )

    def save(self, *args, **kwargs):
        self.file.name = os.path.join(*urlparse(self.s3_file).path.split('/')[2:])

        return super().save(*args, **kwargs)

    class Meta:
        verbose_name = _('Ресурс со SCORM-курсом')
        verbose_name_plural = _('Ресурсы со SCORM-курсом')


@python_2_unicode_compatible
class ResourceTag(models.Model):
    """
    Тэги для ресурсов
    """
    name = models.CharField(_('Тэг'), max_length=255)
    dependent_tags = models.ManyToManyField(
        'self',
        verbose_name=_('Зависимые теги'),
        blank=True,
        symmetrical=False,
        related_name='additional_for',
    )

    class Meta(object):
        verbose_name = _('Тэг для ресурсов')
        verbose_name_plural = _('Тэги для ресурсов')

    def __str__(self):
        """
        Строковое представление тэга
        """
        return str(self.name)
