from builtins import object

from rest_framework import serializers

from kelvin.common.serializer_mixins import DateUpdatedFieldMixin, SkipNullsMixin
from kelvin.common.serializers import DictSerializer
from kelvin.switcher import get_feature_bool_option

from .models import Resource, ResourceTag


class ResourceTagSerializer(serializers.ModelSerializer):
    """
    Сериализатор для тэгов ресурсов
    """
    class Meta(object):
        model = ResourceTag


class BaseResourceSerializer(
    SkipNullsMixin,
    DateUpdatedFieldMixin,
    serializers.ModelSerializer
):
    """
        Сериализатор модели ресурсов
        """
    type = serializers.CharField(source='get_type', read_only=True)

    class Meta(object):
        model = Resource
        fields = (
            'id',
            'date_updated',
            'file',
            'type',
            'name',
        )
        read_only_fields = (
            'date_updated',
            'type',
        )
        list_serializer_class = DictSerializer


class ResourceSerializerCorp(BaseResourceSerializer):
    """
    Сериализатор модели ресурсов, специфичный для corp
    """
    class Meta(BaseResourceSerializer.Meta):
        model = Resource
        fields = BaseResourceSerializer.Meta.fields + (
            'nda',
            'shortened_file_url',
        )
        read_only_fields = BaseResourceSerializer.Meta.read_only_fields + (
            'shortened_file_url',
        )

    def to_representation(self, instance):
        ret = super(ResourceSerializerCorp, self).to_representation(instance)

        feature = 'E7N_ENABLE_NEW_RESOURCE_URLS'

        if get_feature_bool_option(feature, False) and ret['shortened_file_url']:
            ret['file'] = ret['shortened_file_url']

        return ret
