import logging
import os.path
import tempfile
import zipfile

import jsonschema
import xmltodict
from PIL import Image

from django.db import transaction

from kelvin.common.s3.s3_boto_client import s3_boto_client

from .manifest_schema import MANIFEST_SCHEMA
from .models import ScormResource
from .settings import SCORM_MANIFEST_FILENAME

log = logging.getLogger(__name__)


def set_image_dimensions(sender, instance, **kwargs):
    """
    Отслеживает изменение файла изображения
    """
    if instance.tracker.has_changed('file'):
        filename, ext = os.path.splitext(instance.file.name)
        if ext not in ('.jpg', '.jpeg', '.png', '.gif'):
            return

        try:
            img = Image.open(instance.file)
            instance.image_width, instance.image_height = img.size
        except IOError:
            instance.image_width = None
            instance.image_height = None


class ScormValidationError(Exception):
    pass


def normalize_item(item):
    if not item or not isinstance(item, dict):
        return
    if 'item' in item:
        subitems = item['item']
        if isinstance(subitems, dict) and 'item' in subitems:
            subitems = [subitems]
        if isinstance(subitems, list):
            del item['item']
            item['items'] = subitems
            for subitem in subitems:
                normalize_item(subitem)


def normalize_manifest(manifest):
    resources = manifest["manifest"]["resources"]["resource"]
    if isinstance(resources, dict):
        resources = [resources]
    manifest["manifest"]["resources"]["resource"] = resources

    organizations = manifest["manifest"]["organizations"]["organization"]
    if isinstance(organizations, dict):
        organizations = [organizations]
    manifest["manifest"]["organizations"]["organization"] = organizations

    for organization in manifest["manifest"]["organizations"]["organization"]:
        normalize_item(organization)


def walk_items(item):
    if isinstance(item, dict):
        if 'item' in item:
            yield item['item']
        if 'items' in item:
            for subitem in item['items']:
                yield from walk_items(subitem)


def prepare_scorm(scorm_resource_id: int, force: bool = False) -> None:
    """
    Подготовка scorm-курса к использованию:
    1. Скачивание zip-архива со scorm-курсом с s3
    2. Распаковка архива
    3. Парсинг и валидация манифеста
    4. Заливка распакованного архива со scorm-курсом на s3
    :param scorm_resource_id: идентификатор ScormResource
    :param force: повторная обработка
    """

    with transaction.atomic():
        # Находим scorm
        scorm = ScormResource.objects.select_for_update().get(id=scorm_resource_id)

        if not scorm.file or (not force and scorm.status != ScormResource.SCORM_MODULE_STATUS_PENDING):
            return

        # Создаем временные папки для скачаннного и распакованнного zip-архива со scorm-курсом
        with tempfile.TemporaryDirectory() as tmp_zip_folder:
            with tempfile.TemporaryDirectory() as tmp_data_folder:
                local_zip_folder = tmp_zip_folder  # временная папка для скачивания zip
                local_data_folder = tmp_data_folder  # временная папка для распаковки zip
                zip_s3_full_path = scorm.file.name  # полный путь к zip на s3
                (
                    zip_s3_folder,  # папка с zip на s3
                    zip_filename,  # имя zip-архива
                ) = os.path.split(zip_s3_full_path)
                zip_local_full_path = os.path.join(local_zip_folder, zip_filename)  # полный путь к zip локальный
                # полный путь к распакованному zip локальный
                local_data_path = os.path.join(local_data_folder, zip_filename)
                # полный путь к распакованному zip на s3
                s3_data_path = os.path.join('scorm', str(scorm_resource_id), zip_filename[:-4])

                # Скачиваем архив со scorm-курсом
                if not s3_boto_client.file_exists(zip_s3_full_path):
                    raise ScormValidationError("File %s not found in s3" % zip_s3_full_path)

                s3_boto_client.download_file(
                    s3_file_path=zip_s3_full_path,
                    local_file_path=zip_local_full_path,
                )

                # Распаковываем архив
                with open(zip_local_full_path, 'rb') as stream:
                    with zipfile.ZipFile(stream, 'r') as zip_stream:
                        zip_stream.extractall(local_data_path)

                # Локальный путь к манифесту
                xml_manifest_path = os.path.join(local_data_path, SCORM_MANIFEST_FILENAME)

                if not os.path.exists(xml_manifest_path):
                    raise ScormValidationError('File %s not found' % SCORM_MANIFEST_FILENAME)

                # Открываем манифест и парсим его в json
                with open(xml_manifest_path, 'r') as xml_manifest_file:
                    xml_manifest = xml_manifest_file.read()
                    json_manifest = xmltodict.parse(xml_manifest)

                # нормализация json-манифест
                normalize_manifest(json_manifest)

                # Валидируем json-манифест по схеме
                jsonschema.validate(json_manifest, MANIFEST_SCHEMA)

                # Проверяем, что в ресурсах нет дубликатов
                # И для каждого ресурса существует файл с точкой входа
                resources_ids = set()
                resources = json_manifest["manifest"]["resources"]["resource"]
                for resource in resources:
                    scorm_type = resource.get('@adlcp:scormType', 'sco')
                    if scorm_type == 'asset':
                        continue
                    resource_id = resource['@identifier']
                    if resource_id in resources_ids:
                        raise ScormValidationError("Resource id %s is not unique" % resource_id)
                    resources_ids.add(resource_id)

                    href = resource['@href']
                    resource_start_file_path = f'{local_data_path}/{href}'

                    if not os.path.exists(resource_start_file_path):
                        raise ScormValidationError(
                            "Start file %s for resource %s not found" % (
                                href,
                                resource_id,
                            ),
                        )

                # Проверяем, что существует дефолтная организация
                # Проверяем, что для каждого листового item существует соответствующий scorm-ресурс
                organizations = json_manifest["manifest"]["organizations"]["organization"]
                default_organization_id = json_manifest["manifest"]["organizations"]["@default"]
                has_default_organization = False
                for organization in organizations:
                    organization_id = organization['@identifier']
                    if organization_id == default_organization_id:
                        has_default_organization = True
                    for leaf_item in walk_items(organization):
                        item_id = leaf_item['@identifier']
                        resource_id = leaf_item['@identifierref']
                        if resource_id not in resources_ids:
                            raise ScormValidationError(
                                'In organization %s in item %s resource %s not found' % (
                                    organization_id,
                                    item_id,
                                    resource_id,
                                ),
                            )

                if not has_default_organization:
                    raise ScormValidationError("No default organization %s" % default_organization_id)

                # Манифест проверен
                # Загружаем распакованный scorm-курс на s3
                s3_boto_client.upload_folder(
                    local_root_path=local_data_path,
                    s3_root_path=s3_data_path,
                )

                # Ставим отметку, что SCORM готов к использованию
                ScormResource.objects.filter(id=scorm_resource_id).update(
                    public_url=s3_data_path,
                    status=ScormResource.SCORM_MODULE_STATUS_READY,
                    error_messages="",
                )
