import json
from builtins import object

from django import forms
from django.conf.urls import url
from django.contrib import admin
from django.core.exceptions import PermissionDenied
from django.core.urlresolvers import reverse
from django.forms import widgets
from django.shortcuts import redirect
from django.template.loader import render_to_string
from django.utils.translation import ugettext_lazy as _

from kelvin.common.admin import build_dropdown_filter_with_title
from kelvin.result_stats.models import (
    CourseLessonStat, DiagnosticsStat, LessonDiagnosticStats, ProblemAnswerStat, ProblemStat, StudentCourseStat,
    StudentDiagnosticsStat,
)
from kelvin.result_stats.tasks import (
    recalculate_courselessonstat, recalculate_problem_answers_stat, recalculate_problemstat,
    recalculate_studentcoursestat,
)


class CourseLessonStatAdmin(admin.ModelAdmin):
    """
    Админка для результатов группы по занятию
    """
    list_display = (
        'clesson',
        'percent_complete',
        'percent_fail',
        'results_count',
        'max_results_count',
        'average_points',
        'average_max_points',
    )
    actions = ('calculate_clesson_stat', )
    search_fields = ('id', 'clesson__id', )

    def calculate_clesson_stat(self, request, queryset):
        """
        Запускает подсчет статистики для записи в отдельном селери-таске
        """
        for clesson_stat in queryset:
            recalculate_courselessonstat.delay(clesson_stat.clesson_id)
    calculate_clesson_stat.short_description = _(u'Пересчитать статистику')


class MarkerStatsWidget(widgets.Widget):
    """
    Виджет для отображения разметки задачи
    """
    RENDER_TEMPLATE = 'admin/result_stats/marker_stat_widget.html'

    def render(self, name, value, attrs=None):
        return render_to_string(
            self.RENDER_TEMPLATE,
            context={
                'marker_stats': (value or []),
            },
        )


class ProblemStatForm(forms.ModelForm):
    """
    Форма статистики
    """
    marker_stats = forms.Field(
        label=_(u'Ответы на маркеры'),
        widget=MarkerStatsWidget,
        required=False,
    )

    class Meta(object):
        model = ProblemStat
        fields = (
            'problem',
            'correct_number',
            'correct_percent',
            'incorrect_number',
            'marker_stats',
        )


class ProblemStatAdmin(admin.ModelAdmin):
    """
    Админка для статистики по задаче
    """
    list_display = (
        'id',
        'problem',
        'get_problem_text',
        'correct_number',
        'incorrect_number',
        'correct_percent',
        'date_updated',
    )
    fields = (
        'problem_edit_link',
        'get_problem_full_text',
        'correct_number',
        'correct_percent',
        'incorrect_number',
        'marker_stats',
    )
    readonly_fields = (
        'problem_edit_link',
        'get_problem_full_text',
        'correct_number',
        'incorrect_number',
        'correct_percent',
    )
    raw_id_fields = ('problem',)
    search_fields = ('id', 'problem__id')
    actions = ('calculate_problem_stat', )

    change_form_template = 'admin/result_stats/problemstat_change_form.html'

    form = ProblemStatForm

    def get_queryset(self, request):
        """
        Добавляет задачу к кверисету
        """
        qs = super(ProblemStatAdmin, self).get_queryset(request)
        return qs.select_related('problem')

    def get_urls(self):
        """
        Добавляет адрес для создания статистики
        """
        info = self.model._meta.app_label, self.model._meta.model_name
        create_url = url(r'^(.+)/create/$',
                         self.admin_site.admin_view(self.create_view),
                         name='{0}_{1}_create'.format(*info))
        return [create_url] + super(ProblemStatAdmin, self).get_urls()

    def calculate_problem_stat(self, request, queryset):
        """
        Запускает подсчет статистики для каждой задачи в отдельном селери-таске
        """
        for problem_stat in queryset:
            recalculate_problemstat.delay(problem_stat.problem_id)
            recalculate_problem_answers_stat.delay(problem_stat.problem_id)
    calculate_problem_stat.short_description = _(u'Пересчитать статистику')

    def problem_edit_link(self, problem_stat):
        """
        Ссылка на задачу
        """
        link = reverse('admin:problems_problem_change',
                       args=(problem_stat.problem_id, ))
        return u'<a href="{0}" target="_blank">{1}</a>'.format(
            link, problem_stat.problem)
    problem_edit_link.short_description = _(u'Задача')
    problem_edit_link.allow_tags = True

    def get_problem_text(self, problem_stat):
        """
        Условие задачи (100 символов из `text` элементов лэйаута)
        """
        return u'\n'.join(
            item['content']['text'] if item['kind'] == 'text'
            else item['content']['options'].get('text', '')
            for item in problem_stat.problem.markup['layout']
        )[:100] + u'...'
    get_problem_text.short_description = _(u'Условие задачи')

    def get_problem_full_text(self, problem_stat):
        """
        Полное условие задачи из `text` элементов лэйаута
        """
        return u'\n'.join(
            item['content']['text'] if item['kind'] == 'text'
            else item['content']['options'].get('text', '')
            for item in problem_stat.problem.markup['layout']
        )
    get_problem_full_text.short_description = _(u'Условие задачи')

    def has_add_permission(self, request):
        """
        Не даем создавать статистику вручную
        """
        return False

    def save_model(self, request, problem_stat, form, change):
        """
        Вместо сохранения пересчитываем статистику (без селери)
        """
        if problem_stat.problem_id:
            recalculate_problemstat(problem_stat.problem_id)

    def create_view(self, request, problem_id):
        """
        Создание статистики по идентификатору задачи
        """
        if not self.has_module_permission(request):
            raise PermissionDenied

        recalculate_problemstat(problem_id)
        recalculate_problem_answers_stat(problem_id)

        stat_id = ProblemStat.objects.values_list(
            'id', flat=True).get(problem_id=problem_id)
        return redirect('admin:result_stats_problemstat_change', stat_id)

    def change_view(self, request, object_id, form_url='', extra_context=None):
        """
        Добавляем в контекст поле со статистикой по ответам
        """
        extra_context = extra_context or {}
        extra_context['answer_stats'] = (
            ProblemAnswerStat.objects
            .filter(problem__problemstat=object_id)
        )

        return super(ProblemStatAdmin, self).change_view(
            request, object_id, form_url, extra_context=extra_context,
        )


class ProblemAnswerStatAdmin(admin.ModelAdmin):
    """
    Админка для статистики по ответам на задачи
    """
    list_display = (
        'problem',
        'markers_answer',
        'is_correct',
        'count',
        'percent',
    )
    readonly_fields = (
        'problem',
        'markers_answer',
        'is_correct',
        'count',
        'percent',
    )
    raw_id_fields = (
        'problem',
    )
    search_fields = (
        '=problem__id',
    )

    def get_queryset(self, request):
        """
        Отсеиваем статистику с процентом < 10
        """
        queryset = super(ProblemAnswerStatAdmin, self).get_queryset(request)
        return queryset.filter(percent__gte=10)


class StudentCourseStatAdmin(admin.ModelAdmin):
    """
    Админка для статистики ученика по курсу
    """
    list_display = (
        'id',
        'course',
        'student',
    )
    raw_id_fields = ('student', 'course')
    search_fields = ('student__username', )
    list_filter = [
        ('course__name', build_dropdown_filter_with_title(u'Курс (название)')),
        ('course__id', build_dropdown_filter_with_title(u'Курс (id)')),
    ]
    actions = ('calculate_student_course_stat', )

    def get_queryset(self, request):
        """
        Добавляет курсы и учеников к кверисету
        """
        qs = super(StudentCourseStatAdmin, self).get_queryset(request)
        return qs.select_related('course', 'student')

    def calculate_student_course_stat(self, request, queryset):
        """
        Запускает полный пересчет всех указанных статистик
        """
        for student_course_stat in queryset:
            recalculate_studentcoursestat.delay(student_course_stat.student_id,
                                                student_course_stat.course_id)
    calculate_student_course_stat.short_description = _(
        u'Пересчитать статистику')


class DiagnosticsStatAdmin(admin.ModelAdmin):
    """Админка `DiagnosticStat`"""
    list_display = (
        'id',
        'course',
        'points',
        'count',
    )
    list_filter = [
        ('course__id', build_dropdown_filter_with_title(u'Курс (id)')),
    ]

    raw_id_fields = (
        'course',
    )


class LessonDiagnosticStatsAdmin(admin.ModelAdmin):
    """Админка `LessonDiagnosticStats`"""
    list_display = (
        'id',
        'clesson',
        'average',
    )

    raw_id_fields = (
        'clesson',
    )


class StudentDiagnosticsStatAdmin(admin.ModelAdmin):
    """Админка `StudentDiagnosticsStat`"""
    list_display = (
        'id',
        'student',
        'course',
        'percent',
        'calculated',
        'email_sent',
        'uuid',
    )
    list_filter = [
        ('course__id',
         build_dropdown_filter_with_title(u'Курс (id)')),
        'calculated',
        'email_sent',
    ]

    raw_id_fields = (
        'student',
        'course',
    )

admin.site.register(CourseLessonStat, CourseLessonStatAdmin)
admin.site.register(ProblemStat, ProblemStatAdmin)
admin.site.register(ProblemAnswerStat, ProblemAnswerStatAdmin)
admin.site.register(StudentCourseStat, StudentCourseStatAdmin)
admin.site.register(DiagnosticsStat, DiagnosticsStatAdmin)
admin.site.register(LessonDiagnosticStats, LessonDiagnosticStatsAdmin)
admin.site.register(StudentDiagnosticsStat, StudentDiagnosticsStatAdmin)
