import json
import logging
import uuid
from builtins import object, str
from collections import defaultdict
from copy import copy
from itertools import chain
from urllib.parse import quote

from django.conf import settings
from django.core.validators import MaxValueValidator, MinValueValidator
from django.db import models, transaction
from django.db.models.aggregates import Sum
from django.utils import timezone
from django.utils.translation import ugettext_lazy as _

from rest_framework.exceptions import ValidationError

from kelvin.common.fields import JSONField
from kelvin.common.model_mixins import TimeStampMixin
from kelvin.common.utils import get_percentage, old_round
from kelvin.courses.models import Course, CourseLessonLink, CourseStudent
from kelvin.lesson_assignments.models import LessonAssignment
from kelvin.lessons.models import LessonProblemLink
from kelvin.problems.models import Problem
from kelvin.results.models import CourseLessonResult, LessonResult

logger = logging.getLogger(__name__)


class CourseLessonStat(models.Model):
    """
    Статистика группы учеников по занятию в курсе
    """
    clesson = models.OneToOneField(
        CourseLessonLink,
        verbose_name=_('Занятие курса'),
    )
    percent_complete = models.IntegerField(
        verbose_name=_('Процент выполнения занятия'),
        validators=[
            MinValueValidator(0),
            MaxValueValidator(100),
        ],
        default=0,
    )
    percent_fail = models.IntegerField(
        verbose_name=_('Процент ошибок'),
        validators=[
            MinValueValidator(0),
            MaxValueValidator(100),
        ],
        default=0,
    )
    results_count = models.IntegerField(
        verbose_name=_('Учтено результатов'),
        default=0,
    )
    max_results_count = models.IntegerField(
        verbose_name=_('Возможное кол-во результатов'),
        default=0,
    )
    average_points = models.FloatField(
        verbose_name=_('Среднее значение баллов за занятие по ученикам'),
        default=0,
    )
    average_max_points = models.FloatField(
        verbose_name=_('Средний максимум баллов по ученикам'),
        default=0,
    )

    class Meta(object):
        verbose_name = _('Статистика прохождения группой занятия')
        verbose_name_plural = _('Статистики прохождения группами занятий')

    def calculate(self, allow_unsolved=True):
        """
        Пересчитывает процент выполнения, среднее число баллов.
        Работает только с уже сохраненным объектом
        :param allow_unsolved: разрешить нерешенные задачи (если запретить,
        то все нерешенные будут считаться неправильными)
        :return: tuple(
            <процент успешно выполненных>,
            <процент ошибок>,
            <всего результатов>,
            <возможное кол-во результатов>,
            <средний балл>,
            <средний максимум баллов>
        )
        """
        student_ids = (
            CourseStudent.objects.filter(
                course=self.clesson.course_id
            ).values_list('student_id', flat=True)
        )

        # при отсутствии учеников в курсе не считаем
        if not student_ids:
            return None

        # Получаем последние результаты каждого ученика
        if self.clesson.date_assignment:
            results_by_student = {
                result.summary.student_id: result
                for result in CourseLessonResult.objects.filter(
                    summary__clesson=self.clesson_id,
                    summary__student_id__in=student_ids,
                    date_created__gte=self.clesson.date_assignment,
                ).select_related('summary')
                .order_by('summary__student_id', '-date_updated')
                .distinct('summary__student_id')
            }
        else:
            results_by_student = {}

        if not results_by_student:
            # если нет результатов, нет смысла что-то считать
            return 0, 0, 0, 0, 0, 0

        default_problem_links = (
            CourseLessonResult.get_summarizable_lessonproblemlinks(
                self.clesson.lesson_id)
        )

        # Получаем назначения учеников
        assignments_by_student = {
            assignment.student_id: assignment
            for assignment in LessonAssignment.objects.filter(
                clesson=self.clesson_id,
                student_id__in=student_ids,
            )
        }
        correct_count, fail_count, total_count = 0, 0, 0
        points_sum = 0
        max_points_sum = 0
        nothing_assigned = 0
        for student_id in student_ids:
            assignment = assignments_by_student.get(student_id)
            if assignment:
                problem_links = [
                    link for link in default_problem_links
                    if link.id in assignment.problems
                ]
                if not problem_links:
                    nothing_assigned += 1
            else:
                # При отсутствии назначения считаем, что назначено все
                problem_links = default_problem_links
            total_count += len(problem_links)

            result = results_by_student.get(student_id)
            if result:
                result_summary = result.get_summary(
                    problem_links, result, lesson_scenario=self.clesson,
                    force_show_results=True,
                )
                correct_count += result.get_correct_count(problem_links, result_summary)
                fail_count += result.get_incorrect_count(problem_links, result_summary)
                points_sum += result.points or 0
                max_points_sum += result.max_points or 0

        results_count = len(results_by_student)
        results_len_float = float(results_count)

        def count_percent(number):
            """
            Для заданного числа возвращает кол-во процентов от целого
            """
            return int(old_round(number / float(total_count), 2) * 100) if total_count else 0

        # Если контрольная, то считаем нерешенные (неначатые)
        # задачи как неправильные.
        if self.clesson.mode == self.clesson.CONTROL_WORK_MODE:
            allow_unsolved = False

        percent_complete = count_percent(correct_count)
        percent_fail = count_percent(fail_count) if allow_unsolved else 100 - percent_complete
        average_points = int(old_round(points_sum / results_len_float))
        # average_max_points = int(round(old_div(max_points_sum, results_len_float)))
        average_max_points = int(old_round(max_points_sum / results_len_float))
        max_results_count = len(student_ids) - nothing_assigned

        # TODO: возвращать что-то более читабельное: объект модели или словарь
        return (
            percent_complete,
            percent_fail,
            results_count,
            max_results_count,
            average_points,
            average_max_points,
        )


class ProblemStat(TimeStampMixin, models.Model):
    """
    Статистика всех ответов по задаче
    """
    problem = models.OneToOneField(
        Problem,
        verbose_name=_('Задача'),
    )
    correct_number = models.IntegerField(
        verbose_name=_('К-во верных ответов на задачу'),
        default=0,
    )
    incorrect_number = models.IntegerField(
        verbose_name=_('К-во неверных ответов на задачу'),
        default=0,
    )
    correct_percent = models.IntegerField(
        verbose_name=_('Процент верных ответов'),
        validators=[
            MinValueValidator(0),
            MaxValueValidator(100),
        ],
        default=0,
    )
    marker_stats = JSONField(
        verbose_name=_('Детальная статистика по маркерам'),
    )

    # Маркеры, по которым детализируется отчет
    EVALUATED_MARKERS = {'inline', 'choice'}

    class Meta(object):
        verbose_name = _('Статистика ответов по задаче')
        verbose_name_plural = _('Статистики ответов по задачам')

    def calculate(self):
        """
        Подсчет статистики по всем ответам на задачу. Возвращает количество
        верных и неверных ответов на задачу в целом, процент верных ответов
        и детальную статистику по проверяемым маркерам/инпутам (указаны в
        `EVALUATED_MARKERS`), которые есть в задаче
        Пример детальной статистики:
        [
            ,['1-1', {  # 'маркер-инпут' для инлайна
                'correct_number': 50,
                'incorrect_number': 150,
                'correct_answers': [
                    ('Верный Ответ', 25),
                    ('верный ответ', 15),
                    ('Верный ответ', 10),
                ]
                'correct_without_answer': 0,
                'incorrect_answers': [
                    ('неверный ответ', 100),
                    ('совсем неверный ответ', 47),
                    ('вфыву213', 1),
                    ('у3122ц', 1),
                ],
                'incorrect_without_answer': 1,
                'correct_percent': 25,
            }],
            ['2', {...}],
        }
        """
        # Без задачи работать нельзя
        assert self.problem_id, 'Невозможно посчитать статистику без задачи'

        # Выбираем, с какими маркерами работаем детально, и готовим для них
        # итоговую структуру
        correct_number = 0
        incorrect_number = 0
        marker_stats = {}
        marker_ids = []
        marker_by_id = {}

        for item in self.problem.markup['layout']:
            if (item['kind'] != 'marker' or
                    item['content']['type'] not in self.EVALUATED_MARKERS):
                continue
            if item['content']['type'] == 'inline':
                for input_id in item['content']['options']['inputs']:
                    # для каждого инпута своя статистика
                    stat_key = '{0}-{1}'.format(item['content']['id'],
                                                input_id)
                    marker_stats[stat_key] = {
                        'correct_number': 0,
                        'correct_answers': defaultdict(lambda: 0),
                        'correct_without_answer': 0,
                        'incorrect_number': 0,
                        'incorrect_answers': defaultdict(lambda: 0),
                        'incorrect_without_answer': 0,
                    }
            else:
                marker_stats[str(item['content']['id'])] = {
                    'correct_number': 0,
                    'correct_answers': defaultdict(lambda: 0),
                    'correct_without_answer': 0,
                    'incorrect_number': 0,
                    'incorrect_answers': defaultdict(lambda: 0),
                    'incorrect_without_answer': 0,
                }
            marker_ids.append(str(item['content']['id']))
            marker_by_id[str(item['content']['id'])] = item['content']

        # Получаем ответы (в курсе и вне курса) на задачу
        problem_links = LessonProblemLink.objects.filter(
            problem_id=self.problem.id)
        problem_link_ids = [str(link.id) for link in problem_links]
        lesson_ids = [link.lesson_id for link in problem_links]

        lesson_answers = LessonResult.objects.filter(
            summary__lesson__in=lesson_ids,
        ).values_list('answers', flat=True)

        clesson_answers = CourseLessonResult.objects.filter(
            summary__clesson__lesson__in=lesson_ids,
        ).values_list('answers', flat=True)

        answers_for_problem = []
        for lesson_answer in chain(lesson_answers, clesson_answers):
            for problem_link_id in problem_link_ids:
                if problem_link_id in lesson_answer:
                    answers_for_problem.extend(lesson_answer[problem_link_id])

        # Обрабатываем ответы. Извлекаем корректность/некорректность ответа
        # в целом и на каждый маркер/инпут, а также конкретный ответ на каждый
        # маркер/инпут.
        # Те ответы пользователей, где 'mistakes': 0, добавляем в правильные
        for answer in answers_for_problem:
            if answer['markers'] is None:
                logger.warning('Skipping answer because markers are None '
                               'for problem %s', self.problem_id)
                continue

            if answer['mistakes']:
                incorrect_number += 1
            else:
                correct_number += 1
            for marker_id in marker_ids:
                if marker_id not in (answer.get('markers') or {}):
                    continue

                marker_answer = answer['markers'][marker_id]

                if marker_by_id[marker_id]['type'] == 'inline':
                    # в инлайновом маркере разбираем все инпуты
                    inputs_data = marker_by_id[marker_id]['options']['inputs']
                    for input_id in inputs_data:
                        stat_key = '{0}-{1}'.format(marker_id, input_id)
                        input_stat = marker_stats[stat_key]

                        if not isinstance(marker_answer['answer_status'],
                                          dict):
                            # не переделанные результаты при переделке
                            # маркеров на инлайновые?
                            continue

                        # NB! по статусу ответа на группу инпутов смотрим
                        # правильность ответа на инпут
                        if marker_answer['answer_status'].get(
                                str(inputs_data[input_id]['group'])):
                            input_stat['correct_number'] += 1
                            correct = True
                        else:
                            input_stat['incorrect_number'] += 1
                            correct = False

                        user_answer = marker_answer.get('user_answer', {}).get(
                            input_id)
                        if user_answer is None:
                            if correct:
                                input_stat['correct_without_answer'] += 1
                            else:
                                input_stat['incorrect_without_answer'] += 1
                        else:
                            # Делаем ответ гарантированно хешируемым
                            user_answer = json.dumps(user_answer,
                                                     sort_keys=True)
                            if correct:
                                input_stat['correct_answers'][user_answer] \
                                    += 1
                            else:
                                input_stat['incorrect_answers'][user_answer] \
                                    += 1
                else:
                    # подсчет статистики всех маркеров кроме инлайнового
                    marker_stat = marker_stats[marker_id]

                    if marker_answer['mistakes']:
                        marker_stat['incorrect_number'] += 1
                        correct = False
                    else:
                        marker_stat['correct_number'] += 1
                        correct = True

                    user_answer = marker_answer.get('user_answer')
                    if user_answer is not None:
                        # Делаем ответ гарантированно хешируемым
                        user_answer = json.dumps(user_answer, sort_keys=True)
                        if correct:
                            marker_stat['correct_answers'][user_answer] += 1
                        else:
                            marker_stat['incorrect_answers'][user_answer] += 1
                    else:
                        if correct:
                            marker_stat['correct_without_answer'] += 1
                        else:
                            marker_stat['incorrect_without_answer'] += 1

        # Подсчитываем процент верных ответов на маркеры и задачу в целом,
        # приводим словари ответов к спискам, отсортированным сначала по
        # убыванию частоты, потом по ответу (алфавитно-числовая по возрастанию)
        # Также возвращаем ответы из json-строк в исходные объекты
        def reformat(answer_dict):
            return [
                (json.loads(answer_number[0]), answer_number[1])
                for answer_number in sorted(
                    answer_dict.items(),
                    key=lambda answer_number: (-answer_number[1], answer_number[0]),
                )
            ]

        for marker_id, marker_stat in marker_stats.items():
            total = (marker_stat['correct_number'] +
                     marker_stat['incorrect_number'])
            marker_stat['correct_percent'] = (
                int(round(float(marker_stat['correct_number']) / total * 100))
            ) if total else 0

            marker_stat['correct_answers'] = reformat(
                marker_stat['correct_answers'])
            marker_stat['incorrect_answers'] = reformat(
                marker_stat['incorrect_answers'])

        marker_stats = [
            [marker_id, marker_stat]
            for marker_id, marker_stat
            in sorted(iter(marker_stats.items()),
                      key=lambda marker_id_data: marker_id_data[0])
        ]

        total = correct_number + incorrect_number
        correct_percent = (
            int(round(float(correct_number) / total * 100)) if total else 0)

        return correct_number, incorrect_number, correct_percent, marker_stats


class StudentCourseStat(models.Model):
    """
    Статистика ученика по всем занятиям курса

    Сейчас мы в этой статистике всегда показываем результат ученика даже с
    учетом текущей контрольной, и это сейчас требуется в учительском интерфейсе
    """
    student = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        verbose_name=_('Ученик'),
    )
    course = models.ForeignKey(
        Course,
        verbose_name=_('Курс'),
    )
    points = models.IntegerField(
        verbose_name=_('Очки'),
        default=0,
    )
    problems_correct = models.IntegerField(
        verbose_name=_('Количество верно решенных задач'),
        default=0,
    )
    problems_incorrect = models.IntegerField(
        verbose_name=_('Количество неверно решенных задач'),
        default=0,
    )
    problems_skipped = models.IntegerField(
        verbose_name=_('Количество пропущенных задач'),
        default=0,
    )
    total_efficiency = models.IntegerField(
        verbose_name=_('Эффективность по всем занятиям'),
        validators=[
            MinValueValidator(0),
            MaxValueValidator(100),
        ],
        default=0,
    )
    clesson_data = JSONField(
        verbose_name=_('Детализация по занятиям'),
        default={},
    )
    trainer_last_level = models.IntegerField(
        verbose_name=_('Последний пройденный уровень тренажера'),
        default=0,
    )

    @staticmethod
    def _get_lesson_result_data(result, assignment_list=None):
        """
        Получение результатов прохождения одного занятия. Возвращает словарь
        с ключами 'efficiency', 'points', 'problems_correct'...
        """
        assigned_problem_links = (
            CourseLessonResult.get_summarizable_lessonproblemlinks(
                result.summary.clesson.lesson_id, assignment_list)
        )

        data = dict()
        data['points'] = result.points
        data['max_points'] = result.max_points
        data['efficiency'] = (
            get_percentage(result.points, result.max_points)
            if result.points else 0
        )
        lesson_summary = CourseLessonResult.get_summary(
            assigned_problem_links, result,
            lesson_scenario=result.summary.clesson,
            force_show_results=True,
        )
        data['problems_correct'] = result.get_correct_count(
            assigned_problem_links, lesson_summary)
        data['problems_incorrect'] = result.get_incorrect_count(
            assigned_problem_links, lesson_summary)
        data['problems_skipped'] = (
            len(assigned_problem_links) - data['problems_correct'] - data['problems_incorrect']
        )
        data['progress'] = get_percentage(
            len(result.answers),
            CourseLessonResult.get_all_lessonproblemlinks_count(result.summary.clesson.lesson_id, assignment_list)
        )
        return data

    def _calculate_by_clesson_data(self, clesson_data):
        """
        Подсчет новых значений полей объекта `StudentCourseStat`
        по сводкам результатов из `clesson_data`; сохранение `clesson_data`
        в поле объекта
        """
        self.points = 0
        self.max_points = 0
        self.problems_correct = 0
        self.problems_incorrect = 0
        self.problems_skipped = 0

        for data in clesson_data.values():
            self.points += data['points'] or 0
            self.max_points += data['max_points']
            self.problems_correct += data['problems_correct']
            self.problems_incorrect += data['problems_incorrect']
            self.problems_skipped += data['problems_skipped']

        self.total_efficiency = get_percentage(self.points, self.max_points)
        self.clesson_data = clesson_data

    def _set_zeroes(self):
        """
        Выставление нулевых значений, если при пересчете не оказалось
        занятий в уроке или ответов ученика
        """
        self.points = 0
        self.problems_correct = 0
        self.problems_incorrect = 0
        self.problems_skipped = 0
        self.total_efficiency = 0
        self.clesson_data = {}

    def calculate(self):
        """
        Полный пересчет. Изменяет объект статистики. Если нет занятий в уроке
        или нет ответов ученика - выставляются нулевые значения
        """
        clesson_ids = list(CourseLessonLink.objects.filter(
            course_id=self.course_id).values_list('id', flat=True))

        # Не считаем, если нет занятий
        if not clesson_ids:
            self._set_zeroes()
            return

        student_results = (
            CourseLessonResult.objects.filter(
                summary__student_id=self.student_id,
                summary__clesson__in=clesson_ids)
            .select_related('summary', 'summary__clesson')
        )

        # Не считаем, если нет результатов
        if not student_results:
            self._set_zeroes()
            return

        assignments = LessonAssignment.objects.filter(
            student_id=self.student_id, clesson__in=clesson_ids)

        clesson_result_map = {
            result.summary.clesson_id: result
            for result in student_results
        }
        clesson_assignment_map = {
            assignment.clesson_id: assignment.problems
            for assignment in assignments
        }

        clesson_data = dict()

        for clesson_id in clesson_ids:
            if clesson_id not in clesson_result_map:
                continue
            assignment_list = clesson_assignment_map.get(
                clesson_id)
            clesson_data[str(clesson_id)] = self._get_lesson_result_data(
                clesson_result_map[clesson_id],
                assignment_list,
            )

        self._calculate_by_clesson_data(clesson_data)

    def recalculate_by_result(self, clesson_result):
        """
        Пересчет статистики по одному обновленному результату
        """
        assignment_list = LessonAssignment.get_student_problems(
            self.student_id,
            clesson_result.summary.clesson_id,
        )
        clesson_data = copy(self.clesson_data)
        result_data = self._get_lesson_result_data(clesson_result, assignment_list)
        clesson_data[str(clesson_result.summary.clesson_id)] = result_data
        self._calculate_by_clesson_data(clesson_data)

    # TODO: надо понять, что делать при удалении clesson'ов.
    # Оно иногда происходит без сигналов

    class Meta(object):
        verbose_name = _('Статистика ученика по курсу')
        verbose_name_plural = _('Статистика учеников по курсам')
        unique_together = ('student', 'course')


class DiagnosticsStat(TimeStampMixin, models.Model):
    """
    Число учеников, набравших определенный балл в диагностике
    """
    course = models.ForeignKey(
        Course,
        verbose_name=_('Курс диагностики'),
    )
    points = models.IntegerField(
        verbose_name=_('Число баллов'),
    )
    count = models.IntegerField(
        verbose_name=_('Число учеников'),
        default=0,
    )

    class Meta(object):
        verbose_name = _('Статистика баллов в диагностике')
        verbose_name_plural = _('Статистики баллов в диагностиках')
        unique_together = ('course', 'points')


class LessonDiagnosticStats(TimeStampMixin, models.Model):
    """
    Средний процент набранных баллов в занятии диагностики
    """
    clesson = models.OneToOneField(
        to=CourseLessonLink,
        verbose_name=_('Занятие диагностики'),
    )
    average = models.IntegerField(
        verbose_name=_('Средний процент')
    )

    class Meta(object):
        verbose_name = _('Статистика по занятию в диагностике')
        verbose_name_plural = _('Статистики по занятиям в диагностиках')


class StudentDiagnosticsStat(TimeStampMixin, models.Model):
    """
    Общий результат ученика по диагностике
    """
    student = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        verbose_name=_('Ученик')
    )
    course = models.ForeignKey(
        Course,
        verbose_name=_('Курс диагностики'),
    )
    percent = models.IntegerField(
        verbose_name=_('Процентиль баллов ученика среди всех учеников'),
        blank=True,
        null=True,
    )
    result = JSONField(
        verbose_name=_('Результаты ученика по каждому занятию'),
        blank=True,
        null=True,
    )
    calculated = models.BooleanField(
        verbose_name=_('Статистика посчитана'),
        default=False,
    )
    email_sent = models.BooleanField(
        verbose_name=_('БЫло послано письмо о результатах диагностики'),
        default=False,
    )
    uuid = models.UUIDField(
        verbose_name=_('Код для общедоступной ссылки'),
        db_index=True,
        default=uuid.uuid4,
        editable=False,
    )

    class Meta(object):
        unique_together = ('student', 'course')
        verbose_name = _('Статистика ученика по диагностике')
        verbose_name_plural = _('Статистики учеников по диагностикам')

    def calculate(self):
        """
        Подсчитывает и обновляет статистику
        """
        clessons = self.course.courselessonlink_set.filter(
            mode=CourseLessonLink.DIAGNOSTICS_MODE,
        ).select_related('lesson', 'diagnostics_text')

        results = CourseLessonResult.objects.filter(
            summary__clesson__in=clessons,
            summary__student_id=self.student_id,
        ).select_related('summary').order_by('summary__clesson__order')
        clesson_by_id = {clesson.id: clesson for clesson in clessons}
        result_by_clesson_id = {result.summary.clesson_id: result
                                for result in results}

        # проверяем, что все занятия завершены и в незавершенных занятиях все
        # результаты завершены
        now = timezone.now()
        for clesson in clessons:
            if clesson.finish_date <= now:
                continue
            result = result_by_clesson_id.get(clesson.id)
            if not (result and (result.completed or result.time_expired(
                    clesson_by_id[result.summary.clesson_id]))):
                raise ValidationError(detail='complete all clessons')

        # считаем данные
        self.result = []
        for clesson in clessons:
            result = result_by_clesson_id.get(clesson.id)
            percent = (
                int(round(float(result.points) / result.max_points * 100))
                if result and result.max_points else None
            )
            self.result.append({
                'clesson_id': clesson.id,
                'name': clesson.lesson.name,
                'percent': percent,
                'text': (
                    clesson.diagnostics_text.render({
                        'user': self.student,
                        'percent': percent,
                        'has_result': bool(result),
                    })
                ),
            })

        # находим процентиль результата ученика
        points = sum(result.points for result in results)
        total_count = (DiagnosticsStat.objects.filter(course=self.course)
                       .aggregate(Sum('count'))['count__sum'])
        if total_count:
            self.percent = get_percentage(
                DiagnosticsStat.objects
                .filter(course=self.course, points__lte=points)
                # если нет статистики "не больше", то считаем самого ученика
                .aggregate(Sum('count'))['count__sum'] or 1,
                total_count,
            )
        else:
            # если нет никакой статистики, то проставляем 100 процентов
            self.percent = 100

        # обновляем статистику
        self.calculated = True
        self.save()


class ProblemAnswerStat(models.Model):
    """
    Статистика по ответу на задачу
    """
    problem = models.ForeignKey(
        Problem,
        verbose_name=_('Задача'),
    )
    markers_answer = JSONField(
        verbose_name=_('Ответ'),
        blank=True,
        null=True,
    )
    is_correct = models.BooleanField(
        verbose_name=_('Верный ответ'),
    )
    count = models.IntegerField(
        verbose_name=_('Количество ответов'),
    )
    percent = models.IntegerField(
        verbose_name=_('Процент ответов'),
        validators=[
            MinValueValidator(0),
            MaxValueValidator(100),
        ],
    )

    class Meta(object):
        verbose_name = _('Статистика по ответу на задачу')
        verbose_name_plural = _('Статистики по ответам на задачу')

    @property
    def front_url(self):
        url = (
            '{front_url}platform/lab/problems/{problem_id}/?answer={answer}'
            .format(
                front_url=settings.FRONTEND_HOST,
                problem_id=self.problem_id,
                answer=quote(json.dumps(self.markers_answer, ensure_ascii=False).encode('utf-8')),
            )
        )
        return url if len(url) <= 4000 else None

    @classmethod
    @transaction.atomic
    def calculate_stats(cls, problem_id):
        """
        Подсчет статистики по всем ответам для задачи
        """
        answers_for_problem = cls.get_problem_answers(problem_id)
        total_answers_count = len(answers_for_problem)

        stats_by_answer = {}
        # загружаем уже существующую статистику
        for answer_stat in ProblemAnswerStat.objects.filter(
                problem=problem_id):
            answer_key = repr(answer_stat.markers_answer)
            if answer_key in stats_by_answer:
                # удаляем дубликат
                answer_stat.delete()
            else:
                answer_stat.count = 0
                stats_by_answer[answer_key] = answer_stat

        for answer in answers_for_problem:
            answer_stat = cls.get_answer_stat(problem_id, stats_by_answer,
                                              answer['markers'])
            answer_stat.count += 1
            answer_stat.is_correct = answer['mistakes'] == 0

        # пересчитываем проценты и сохраняем
        for answer_stat in stats_by_answer.values():
            if answer_stat.count == 0:
                # похоже, что статистика не нужна
                answer_stat.delete()
            else:
                answer_stat.percent = int(
                    old_round(float(answer_stat.count) / total_answers_count * 100)
                ) if total_answers_count else 0
                answer_stat.save()

    @staticmethod
    def get_problem_answers(problem_id):
        """
        Возвращает все ответы на задачу
        """
        problem_links = (
            LessonProblemLink.objects
            .filter(problem_id=problem_id)
            .values('id', 'lesson_id')
        )
        problem_link_ids = [str(link['id']) for link in problem_links]
        lesson_ids = [link['lesson_id'] for link in problem_links]

        lesson_answers = LessonResult.objects.filter(
            summary__lesson__in=lesson_ids,
        ).values_list('answers', flat=True)
        clesson_answers = CourseLessonResult.objects.filter(
            summary__clesson__lesson__in=lesson_ids,
        ).values_list('answers', flat=True)
        answers_for_problem = []
        for lesson_answer in chain(lesson_answers, clesson_answers):
            for problem_link_id in problem_link_ids:
                if problem_link_id in lesson_answer:
                    answers_for_problem.extend(lesson_answer[problem_link_id])
        return answers_for_problem

    @classmethod
    def get_answer_stat(cls, problem_id, stats_by_answer, markers_answer):
        """
        Находим статистику для данного ответа либо создаем новую и добавляем
        в `stats_by_answer`
        """
        markers_answer = cls.clear_answer_fields(markers_answer)
        answer_key = repr(markers_answer)
        if answer_key in stats_by_answer:
            return stats_by_answer[answer_key]
        new_stat = ProblemAnswerStat(
            problem_id=problem_id,
            markers_answer=markers_answer,
            count=0,
        )
        stats_by_answer[answer_key] = new_stat
        return new_stat

    @staticmethod
    def clear_answer_fields(markers_answer):
        """
        Удаляет из ответов на маркеры все поля кроме `user_answer`
        """
        return {
            marker_id: {'user_answer': marker_data['user_answer']}
            if 'user_answer' in marker_data else {}
            for marker_id, marker_data in markers_answer.items()
        }
