from builtins import object, str

from rest_framework import fields, serializers

from kelvin.accounts.serializers import BaseUserSerializer
from kelvin.courses.serializers import CourseInEventSerializer
from kelvin.result_stats.models import CourseLessonStat, StudentCourseStat, StudentDiagnosticsStat


class StudentCourseStatInReportSerializer(serializers.ModelSerializer):
    """
    Сериализатор статистики ученика по курсу для курсового журнала
    """
    lessons = fields.SerializerMethodField()

    def get_lessons(self, stat):
        """
        Список эффективности по каждому занятию, с учетом порядка занятий
        в курсе
        """
        return {
            clesson_id: stat.clesson_data.get(str(clesson_id), {}).get('efficiency')
            for clesson_id in self.context['clessons']
        }

    class Meta(object):
        model = StudentCourseStat
        fields = (
            'lessons',
            'points',
            'problems_correct',
            'problems_incorrect',
            'problems_skipped',
            'total_efficiency',
        )


class StudentCourseStatInJournalSerializer(serializers.ModelSerializer):
    """
    Сериализатор статистики ученика по курсу для курсового журнала
    """
    lessons = fields.SerializerMethodField()

    def get_lessons(self, stat):
        """
        Список эффективности по каждому занятию, с учетом порядка занятий
        в курсе
        """
        return {
            clesson_id: dict(
                progress=stat.clesson_data.get(str(clesson_id), {}).get('progress'),
                points=stat.clesson_data.get(str(clesson_id), {}).get('points'),
                max_points=stat.clesson_data.get(str(clesson_id), {}).get('max_points'),
            )
            for clesson_id in self.context['clessons']
        }

    class Meta(object):
        model = StudentCourseStat
        fields = (
            'lessons',
            'points',
            'problems_correct',
            'problems_incorrect',
            'problems_skipped',
            'total_efficiency',
            'student_id',
        )


class StudentCourseStatShortSerializer(serializers.ModelSerializer):
    """
    Сериализатор статистики ученика по курсу для сводки по ученикам
    """
    LAST_LESSONS_COUNT = 5

    lessons = fields.SerializerMethodField()

    def get_lessons(self, obj):
        """
        Список эффективности по каждому занятию, с учетом порядка занятий
        в курсе
        """
        lessons_efficiency = []
        for clesson_id in reversed(self.context['order']):
            clesson_data = obj.clesson_data.get(str(clesson_id))
            if clesson_data:
                lessons_efficiency.append(clesson_data['efficiency'])
                if len(lessons_efficiency) == self.LAST_LESSONS_COUNT:
                    break
        return lessons_efficiency[::-1]

    class Meta(object):
        model = StudentCourseStat
        fields = (
            'lessons',
            'total_efficiency',
        )


class CourseLessonStatSerializer(serializers.ModelSerializer):
    """
    Сериализатор статистики по занятию в курсе
    """

    class Meta(object):
        model = CourseLessonStat
        fields = (
            'percent_complete',
            'percent_fail',
            'results_count',
            'max_results_count',
        )


class StudentDiagnosticsStatSerializers(serializers.ModelSerializer):
    """
    Сериализатор статистики ученика по диагностике
    """
    course = CourseInEventSerializer()
    student = BaseUserSerializer()

    class Meta(object):
        model = StudentDiagnosticsStat
        fields = (
            'id',
            'course',
            'percent',
            'result',
            'student',
        )
