from collections import defaultdict
from datetime import timedelta

from past.utils import old_div

from django.conf import settings
from django.db import transaction
from django.utils import timezone

from rest_framework.exceptions import ValidationError

from kelvin.celery import app
from kelvin.common.decorators.celery_throttler import throttled_task
from kelvin.common.decorators.task_logger import logged_task
from kelvin.common.utils import get_percentage
from kelvin.courses.journal import CourseGroupJournal, LessonJournal
from kelvin.courses.models import Course, CourseLessonLink, CourseStudent
from kelvin.lessons.models import LessonProblemLink
from kelvin.result_stats.emails import send_diagnostics_result_email
from kelvin.result_stats.models import (
    CourseLessonStat, DiagnosticsStat, LessonDiagnosticStats, ProblemAnswerStat, ProblemStat, StudentCourseStat,
    StudentDiagnosticsStat,
)
from kelvin.result_stats.utils import save_journal_table_to_model
from kelvin.results.models import CourseLessonResult, LessonResult

MINUTE = 60


@logged_task
@throttled_task(acks_late=True)
def calculate_lesson_journal(clesson_id):
    """
    Считает статистику по занятию и сохраняет в .csv файл удаляя предыдущий
    При большом количестве учеников процесс долгий, так что таск
    запускается только если учеников в занятии >=
    MIN_LESSON_CSV_STUDENTS
    """

    clesson = CourseLessonLink.objects.get(id=clesson_id)
    last_clesson_result = (
        CourseLessonResult.objects
        .filter(summary__clesson_id=clesson.pk)
        .order_by('-date_updated')
        .first()
    )
    if not last_clesson_result or clesson.date_updated >= last_clesson_result.date_updated:
        return

    count = CourseStudent.objects.filter(course_id=clesson.course_id).count()
    if count < settings.MIN_LESSON_CSV_STUDENTS:
        return

    table = LessonJournal(clesson).table()
    save_journal_table_to_model(table, clesson)


@logged_task
@throttled_task(acks_late=True, soft_time_limit=MINUTE * 10, time_limit=MINUTE * 20)
def calculate_course_journal(course_id):
    """
    Считает статистику по курсу и сохраняет в .csv файл удаляя предыдущий
    При большом количестве учеников процесс долгий, так что таск
    запускается только если учеников в курсе >=
    MIN_COURSE_CSV_STUDENTS
    """
    course = Course.objects.get(id=course_id)

    last_course_result = (
        CourseLessonResult.objects
        .filter(summary__clesson__course_id=course.pk)
        .order_by('-date_updated')
        .first()
    )
    if not last_course_result or course.date_updated >= last_course_result.date_updated:
        return

    count = CourseStudent.objects.filter(course=course).count()
    if count < settings.MIN_COURSE_CSV_STUDENTS:
        return

    table = CourseGroupJournal(course).table()
    save_journal_table_to_model(table, course)


@logged_task
@app.task()
def recalculate_courselessonstat(clesson_id):
    """
    Пересчитать статистику группы по занятию
    """
    stat, created = CourseLessonStat.objects.get_or_create(
        clesson_id=clesson_id)
    result = stat.calculate()
    if result is None:
        stat.delete()
    else:
        (percent_complete, percent_fail,
         results_count, max_results_count,
         avg_points, avg_max) = result
        stat.percent_complete = percent_complete
        stat.percent_fail = percent_fail
        stat.results_count = results_count
        stat.max_results_count = max_results_count
        stat.average_points = avg_points
        stat.average_max_points = avg_max
        stat.save()


@logged_task
@app.task()
def recalculate_problemstat(problem_id):
    """
    Пересчет статистики по задаче
    """
    stat, _ = ProblemStat.objects.get_or_create(
        problem_id=problem_id,
        defaults=dict(marker_stats=[]),
    )

    correct_number, incorrect_number, correct_percent, marker_stats = (stat.calculate())
    stat.correct_number = correct_number
    stat.incorrect_number = incorrect_number
    stat.correct_percent = correct_percent
    stat.marker_stats = marker_stats
    stat.save()


@logged_task
@app.task()
def recalculate_problem_answers_stat(problem_id):
    """
    Пересчет статистики по ответам на задачу
    """
    ProblemAnswerStat.calculate_stats(problem_id)


@logged_task
@app.task()
def daily_recalculate_problemstat():
    """
    Ищем задачи, которые участвуют в измененных результатах за последний день и
    пересчитываем статистику по этим задачам
    """
    yesterday = timezone.now() - timedelta(days=1)
    lesson_ids = set(
        CourseLessonResult.objects
        .filter(date_updated__gte=yesterday)
        .select_related('summary', 'summary__clesson',
                        'summary__clesson__lesson')
        .values_list('summary__clesson__lesson', flat=True)
        .distinct()
    )
    lesson_ids.update(
        LessonResult.objects
        .filter(date_updated__gte=yesterday)
        .select_related('summary', 'summary__lesson')
        .values_list('summary__lesson', flat=True)
        .distinct()
    )
    for problem_id in (LessonProblemLink.objects
                       .filter(lesson__in=lesson_ids,
                               problem__isnull=False)
                       .values_list('problem', flat=True)
                       .distinct()
                       .order_by()):  # сбрасываем сортировку для distinct

        recalculate_problemstat.delay(problem_id)
        recalculate_problem_answers_stat.delay(problem_id)


@logged_task
@app.task()
def recalculate_studentcoursestat(student_id, course_id):
    """
    Полный пересчет статистики ученика по курсу
    """
    with transaction.atomic():
        stat, created = (
            StudentCourseStat.objects.select_for_update().get_or_create(
                student_id=student_id, course_id=course_id)
        )
        stat.calculate()
        stat.save()


@logged_task
@app.task()
def recalculate_studentcoursestat_by_result(result_id):
    """
    Обновление статистики ученика по одному результату
    Делает полный подсчет, если статистика еще не была создана
    """
    try:
        clesson_result = CourseLessonResult.objects.select_related(
            'summary', 'summary__clesson').get(id=result_id,
                                               summary__student__isnull=False)
    except CourseLessonResult.DoesNotExist:
        return
    with transaction.atomic():
        stat, created = (
            StudentCourseStat.objects.select_for_update().get_or_create(
                student_id=clesson_result.summary.student_id,
                course_id=clesson_result.summary.clesson.course_id,
            )
        )

        # Если объект был только что создан, то делаем полный подсчет
        if created:
            stat.calculate()
        else:
            stat.recalculate_by_result(clesson_result)
        stat.save()


@logged_task
@app.task()
def recalculate_diagnostics_stats():
    """
    Подсчет статистики баллов и занятий в диагностиках, обновление статистики
    ученика в диагностике
    """
    # находим идентификаторы курсов, в которых есть незавершенные диагностики
    # или закрывшиеся за последний час
    now = timezone.now()
    course_ids = set(CourseLessonLink.objects.filter(
        mode=CourseLessonLink.DIAGNOSTICS_MODE,
        finish_date__gte=now - timedelta(hours=1),
    ).values_list('course_id', flat=True))

    if not course_ids:
        # при отсутствии текущих курсов с диагностиками ничего не делаем
        return

    for course_id in course_ids:
        clessons = CourseLessonLink.objects.filter(
            mode=CourseLessonLink.DIAGNOSTICS_MODE,
            course_id=course_id,
        )
        clessons_count = len(clessons)
        clesson_by_id = {clesson.id: clesson for clesson in clessons}
        diagnostics_stats_dict = {
            (stat.course_id, stat.points): stat
            for stat in DiagnosticsStat.objects.filter(course_id=course_id)
        }
        clessons_stats_dict = {
            stat.clesson_id: stat
            for stat in LessonDiagnosticStats.objects.filter(
                clesson__in=clessons)
        }
        results = CourseLessonResult.objects.filter(
            summary__clesson__in=clessons,
            summary__student__isnull=False,
        ).select_related('summary')

        # обнуляем статистику по баллам
        for stat in diagnostics_stats_dict.values():
            stat.count = 0

        # проценты по занятиям
        clesson_percents = {clesson.id: [] for clesson in clessons}

        # баллы по ученикам, в словаре ключ - идентификатор ученика,
        # значение - список баллов по завершенным занятиям
        student_points = defaultdict(list)

        # подсчитываем статистики
        for result in results:
            if (clesson_by_id[result.summary.clesson_id].finish_date > now and not (
                result.completed or result.time_expired(clesson_by_id[result.summary.clesson_id])
            )):
                # занятие и попытка не завершены
                continue

            # баллы для подсчета статистики по баллам
            student_points[result.summary.student_id].append(result.points)

            # подсчитываем статистику по занятиям
            clesson_percents[result.summary.clesson_id].append(
                float(result.points) / result.max_points * 100
                if result.max_points else 0
            )

        # подсчитываем статистику по баллам
        for points_list in student_points.values():
            if len(points_list) != clessons_count:
                # не по всем занятиям у ученика есть результаты
                continue
            points = sum(points_list)
            stat = diagnostics_stats_dict.get((course_id, points))
            if not stat:
                # создаем несуществующую статистику
                stat, __ = DiagnosticsStat.objects.get_or_create(
                    course_id=course_id,
                    points=points,
                )
                stat.count = 0
                diagnostics_stats_dict[(course_id, points)] = stat
            stat.count += 1

        # обновляем статистики по баллам
        for stat in diagnostics_stats_dict.values():
            stat.save()

        # обновляем статистики по занятиям
        for clesson in clessons:
            if clesson_percents[clesson.id]:
                stat = clessons_stats_dict.get(clesson.id)
                if not stat:
                    stat, __ = LessonDiagnosticStats.objects.get_or_create(
                        clesson=clesson,
                        average=0,
                    )
                stat.average = int(round(
                    old_div(sum(clesson_percents[clesson.id]), len(clesson_percents[clesson.id]))
                ))
                stat.save()

        # обновляем статистики учеников, исходя из посчитанной статистики
        total_count = sum(
            stat.count for stat in diagnostics_stats_dict.values()
        )
        if not total_count:
            # если нет статистики по баллам, то и обновлять нечего
            return
        results_dict = {
            (result.summary.student_id, result.summary.clesson_id): result
            for result in results
        }
        student_ids = {result.summary.student_id for result in results}
        course = Course.objects.get(id=course_id)

        for student_id in student_ids:
            # по каждому ученику с результатами проверяем, не пришло ли время
            # создать его результат
            student_result_count = 0
            for clesson in clessons:
                # проверяем, завершено ли занятие или есть ли результат
                if clesson.finish_date > now:
                    student_result_count += 1
                else:
                    result = results_dict.get((student_id, clesson.id))
                    if result and (
                        result.completed or result.time_expired(clesson)
                    ):
                        student_result_count += 1

            if student_result_count != clessons_count:
                # ученик еще не завершил диагностику
                continue

            # баллы ученика
            points = sum(student_points[student_id])

            # создаем или обновляем статистику ученика
            stat, created = (
                StudentDiagnosticsStat.objects.select_related('student')
                .get_or_create(course_id=course_id, student_id=student_id)
            )
            if stat.calculated:
                # число учеников, набравших меньше или столько же баллов
                student_count = sum(
                    stat.count for stat in diagnostics_stats_dict.values()
                    if stat.points <= points
                )

                # посылаем письмо при необходимости
                email_sent = False
                if not stat.email_sent:
                    email_sent = send_diagnostics_result_email(
                        course, stat)

                # обновляем статистику
                StudentDiagnosticsStat.objects.filter(id=stat.id).update(
                    percent=get_percentage(student_count, total_count),
                    # обновляем `sent_email` на `True`, если отправили письмо
                    email_sent=stat.email_sent or email_sent,
                )
            else:
                try:
                    stat.calculate()  # чтобы не дублировать код
                except ValidationError:
                    # не все занятия еще закончены у ученика
                    continue
                email_sent = send_diagnostics_result_email(
                    course, stat)
                if email_sent:
                    StudentDiagnosticsStat.objects.filter(id=stat.id).update(
                        email_sent=True,
                    )
