from django.contrib import admin
from django.db import transaction
from django.utils.translation import ugettext_lazy as _

from kelvin.common.admin import build_dropdown_filter_with_title, superuser_only_action
from kelvin.results.models import (
    CourseLessonResult, CourseLessonResultHook, CourseLessonSummary, LessonResult, LessonSummary,
)
from kelvin.results.services.hooks import send_hook
from kelvin.results.utils import recalculate_clesson_result, recalculate_lesson_result

from .services.reset_results import reset_results


class LessonResultAdmin(admin.ModelAdmin):
    """
    Админка результатов занятия
    """
    date_hierarchy = 'date_updated'
    list_display = (
        'get_lesson',
        'get_lesson_name',
        'get_student',
        'points',
        'completed',
        'date_updated',
        'date_created',
    )
    list_select_related = (
        'summary',
        'summary__lesson',
        'summary__student',
    )
    search_fields = (
        'summary__student__username',
        'summary__lesson__id',
    )
    actions = (
        'recalculate_action',
    )

    def recalculate_action(self, request, queryset):
        """Пересчитывает результаты"""
        for result in queryset.prefetch_related('summary', 'summary__lesson'):
            recalculate_lesson_result(result)
    recalculate_action.short_description = u'Пересчитать результат'

    def get_lesson(self, obj):
        return obj.summary.lesson
    get_lesson.short_description = u'Пройденное занятие'

    def get_lesson_name(self, obj):
        return obj.summary.lesson.name
    get_lesson_name.short_description = u'Урок'

    def get_student(self, obj):
        return obj.summary.student
    get_student.short_description = u'Ученик, прошедший занятие'
    get_student.admin_order_field = 'summary__student__username'


class CourseLessonResultAdmin(admin.ModelAdmin):
    """
    Админка результатов занятия в курсе
    """
    date_hierarchy = 'date_updated'
    list_display = (
        'get_clesson',
        'get_lesson_name',
        'get_student',
        'points',
        'completed',
        'date_updated',
        'date_created',
    )
    raw_id_fields = (
        'summary',
        'meta',
    )
    list_select_related = (
        'summary',
        'summary__clesson',
        'summary__clesson__lesson',
        'summary__clesson__course',
        'summary__student',
    )
    search_fields = (
        'summary__student__username',
        'summary__clesson__id',
    )
    list_filter = (
        (
            'summary__clesson__course__name',
            build_dropdown_filter_with_title(u'Курсы')
        ),
        (
            'summary__clesson__lesson__name',
            build_dropdown_filter_with_title(u'Модули')
        ),
        (
            'summary__student__username',
            build_dropdown_filter_with_title(u'Студенты')
        ),
    )
    actions = (
        'recalculate_action',
        'reset_action',
    )

    def get_actions(self, request):
        if request.user.is_superuser:
            return super(CourseLessonResultAdmin, self).get_actions(request)
        else:
            return []

    @superuser_only_action
    def recalculate_action(self, request, queryset):
        """Пересчитывает результаты"""
        for result in queryset.prefetch_related(
                'summary', 'summary__clesson', 'summary__clesson__lesson'):
            recalculate_clesson_result(result)
    recalculate_action.short_description = u'Пересчитать результат'

    @superuser_only_action
    def reset_action(self, request, queryset):
        """Сбросить результаты"""
        from kelvin.result_stats.tasks import calculate_course_journal, calculate_lesson_journal

        queryset = queryset.select_related('summary', 'summary__clesson')
        deleted_data = []
        course_lessons_to_recalc = set()
        courses_to_recalc = set()
        with transaction.atomic():
            for course_lesson_result in queryset:
                student_id = course_lesson_result.summary.student_id

                course_lesson_id = course_lesson_result.summary.clesson_id
                course_id = course_lesson_result.summary.clesson.course_id

                course_lessons_to_recalc.add(course_lesson_id)
                courses_to_recalc.add(course_id)

                deleted_data_item = reset_results(
                    student_id=student_id,
                    course_lesson_id=course_lesson_id,
                    delete_course_lesson_result=True,
                    recalculate_journals=False,
                )
                deleted_data.append(
                    {
                        'student_id': student_id,
                        'course_id': course_id,
                        'course_lesson_id': course_lesson_id,
                        'deleted_data': deleted_data_item,
                    }
                )

        for course_lesson_to_recalc in course_lessons_to_recalc:
            calculate_lesson_journal.delay(clesson_id=course_lesson_to_recalc)

        for course_to_recalc in courses_to_recalc:
            calculate_course_journal.delay(course_id=course_to_recalc)

        self.message_user(request, u'Удаленные результаты: {}'.format(deleted_data))
    reset_action.short_description = u'Сбросить результат'

    def get_clesson(self, obj):
        return obj.summary.clesson
    get_clesson.short_description = u'Пройденное курсозанятие'

    def get_lesson_name(self, obj):
        return obj.summary.clesson.lesson.name
    get_lesson_name.short_description = u'Урок'

    def get_student(self, obj):
        return obj.summary.student
    get_student.short_description = u'Ученик, прошедший курсозанятие'
    get_student.admin_order_field = 'summary__student__username'


class CourseLessonResultInline(admin.TabularInline):
    """ Инлайновая форма результатов для сводки """
    model = CourseLessonResult
    show_change_link = True
    fields = (
        'id',
        'points',
        'max_points',
        'spent_time',
        'completed',
        'work_out',
    )
    readonly_fields = (
        'id',
    )
    extra = 0


class CourseLessonSummaryAdmin(admin.ModelAdmin):
    """
    Админка результатов занятия в курсе
    """
    date_hierarchy = 'date_updated'
    list_display = (
        'clesson',
        'get_lesson_name',
        'get_student',
        'date_updated',
        'date_created',
    )
    list_select_related = (
        'clesson',
        'clesson__lesson',
        'student',
    )
    search_fields = (
        'student__username',
    )
    list_filter = (
        ('clesson__course__name',
         build_dropdown_filter_with_title(u'Курсы учеников')),
    )
    inlines = (
        CourseLessonResultInline,
    )
    raw_id_fields = (
        'clesson',
        'student',
    )

    def get_lesson_name(self, obj):
        return obj.clesson.lesson.name
    get_lesson_name.short_description = u'Урок'

    def get_student(self, obj):
        return obj.student
    get_student.short_description = u'Ученик, прошедший курсозанятие'
    get_student.admin_order_field = 'student__username'


@admin.register(CourseLessonResultHook)
class CourseLessonResultHookAdmin(admin.ModelAdmin):
    raw_id_fields = (
        'course_lesson_result',
    )
    list_select_related = (
        'course_lesson_result',
    )
    date_hierarchy = 'send_time'
    actions = (
        'send_hook_action',
    )

    def send_hook_action(self, request, queryset):
        for obj in queryset:
            send_hook(course_lesson_result_hook=obj, commit=True)

    send_hook_action.short_description = _('Отправить хук')


admin.site.register(LessonResult, LessonResultAdmin)
admin.site.register(CourseLessonResult, CourseLessonResultAdmin)
admin.site.register(LessonSummary)
admin.site.register(CourseLessonSummary, CourseLessonSummaryAdmin)
