import csv

from django_bulk_update.helper import bulk_update
from tqdm import tqdm

from django.core.management.base import BaseCommand
from django.db import transaction

from kelvin.courses.models import CourseLessonLink, CourseStudent
from kelvin.result_stats.tasks import (
    calculate_course_journal, calculate_lesson_journal, recalculate_courselessonstat, recalculate_studentcoursestat,
)

from ...models import CourseLessonResult, CourseLessonSummary

BULK_BATCH_SIZE_DEFAULT = 1000


class DryRunException(Exception):
    pass


class Command(BaseCommand):
    def move_results(self, options):
        source_course_lesson_id = options['source_course_lesson_id']
        dest_course_lesson_id = options['dest_course_lesson_id']
        conflict_resolve_strategy = options['conflict_resolve_strategy']
        logfile = options['logfile']
        progress = options.get('progress', False)

        try:
            source_course_lesson = CourseLessonLink.objects.get(id=source_course_lesson_id)
        except CourseLessonLink.DoesNotExist:
            self.stderr.print(f'Course lesson {source_course_lesson_id} does not exist')
            return

        try:
            dest_course_lesson = CourseLessonLink.objects.get(id=dest_course_lesson_id)
        except CourseLessonLink.DoesNotExist:
            self.stderr.print(f'Course lesson {dest_course_lesson_id} does not exist')
            return

        try:
            with transaction.atomic():
                source_results = (
                    CourseLessonResult.objects
                    .select_related('summary', 'summary__clesson')
                    .filter(summary__clesson=source_course_lesson)
                )
                existing_dest_results_map = {
                    result.summary.student_id: result
                    for result in (
                        CourseLessonResult.objects
                        .filter(summary__clesson=dest_course_lesson)
                        .select_related('summary', 'summary__clesson')
                    )
                }
                existing_dest_summaries_map = {
                    summary.student_id: summary
                    for summary in (
                        CourseLessonSummary.objects
                        .filter(clesson=dest_course_lesson)
                        .select_related('clesson')
                    )
                }

                existing_dest_course_students = {
                    (course_student[0], course_student[1])
                    for course_student
                    in (
                        CourseStudent.objects
                        .filter(course_id=dest_course_lesson.course_id)
                        .values_list('course_id', 'student_id')
                    )
                }

                summaries_to_create = []
                results_to_create = []
                results_to_update = []
                course_students = []
                course_students_to_create = []

                with open(logfile, 'w') as csv_file:
                    fieldnames = [
                        'result_action',
                        'summary_action',
                        'student_id',
                        'source_result_id',
                        'source_summary_id',
                        'source_answers',
                        'source_points',
                        'source_max_points',
                        'source_spent_time',
                        'source_completed',
                        'source_work_out',
                        'source_viewed',
                        'dest_result_id',
                        'dest_summary_id',
                        'dest_answers',
                        'dest_points',
                        'dest_max_points',
                        'dest_spent_time',
                        'dest_completed',
                        'dest_work_out',
                        'dest_viewed',
                    ]

                    csv_writer = csv.DictWriter(csv_file, fieldnames=fieldnames)
                    csv_writer.writeheader()

                    self.stdout.write('Copy results:')
                    if progress:
                        count_source_results = source_results.count()
                        source_results = tqdm(source_results, total=count_source_results)
                    for source_result in source_results:
                        student_id = source_result.summary.student_id
                        if student_id in existing_dest_results_map:
                            dest_result = existing_dest_results_map[student_id]
                            to_update = (
                                (
                                    conflict_resolve_strategy == 'min' and
                                    source_result.points < dest_result.points
                                ) or
                                (
                                    conflict_resolve_strategy == 'max' and
                                    source_result.points > dest_result.points
                                )
                            )
                            if to_update:
                                log_row = {
                                    'result_action': 'update',
                                    'summary_action': 'update',
                                    'student_id': student_id,
                                    'source_result_id': source_result.id,
                                    'source_summary_id': source_result.summary_id,
                                    'source_answers': source_result.answers,
                                    'source_points': source_result.points,
                                    'source_max_points': source_result.max_points,
                                    'source_spent_time': source_result.spent_time,
                                    'source_completed': source_result.completed,
                                    'source_work_out': source_result.work_out,
                                    'source_viewed': source_result.viewed,
                                    'dest_result_id': dest_result.id,
                                    'dest_summary_id': dest_result.summary_id,
                                    'dest_answers': dest_result.answers,
                                    'dest_points': dest_result.points,
                                    'dest_max_points': dest_result.max_points,
                                    'dest_spent_time': dest_result.spent_time,
                                    'dest_completed': dest_result.completed,
                                    'dest_work_out': dest_result.work_out ,
                                    'dest_viewed': dest_result.viewed,
                                }
                                csv_writer.writerow(log_row)

                                dest_result.answers = source_result.answers
                                dest_result.points = source_result.points
                                dest_result.max_points = source_result.max_points
                                dest_result.spent_time = source_result.spent_time
                                dest_result.completed = source_result.completed
                                dest_result.work_out = source_result.work_out
                                dest_result.viewed = source_result.viewed
                                results_to_update.append(dest_result)

                        else:
                            if student_id in existing_dest_summaries_map:
                                summary = existing_dest_summaries_map[student_id]
                            else:
                                summary_to_create = CourseLessonSummary(
                                    clesson=dest_course_lesson,
                                    student_id=student_id,
                                    lesson_finished=source_result.summary.lesson_finished,
                                )
                                summaries_to_create.append(summary_to_create)
                                summary = summary_to_create
                            result_to_create = CourseLessonResult(
                                answers=source_result.answers,
                                points=source_result.points,
                                max_points=source_result.max_points,
                                spent_time=source_result.spent_time,
                                completed=source_result.completed,
                                work_out=source_result.work_out,
                                viewed=source_result.viewed,
                            )

                            log_row = {
                                'result_action': 'create',
                                'summary_action': 'create' if summary.id is None else 'update',
                                'student_id': student_id,
                                'source_result_id': source_result.id,
                                'source_summary_id': source_result.summary_id,
                                'source_answers': source_result.answers,
                                'source_points': source_result.points,
                                'source_max_points': source_result.max_points,
                                'source_spent_time': source_result.spent_time,
                                'source_completed': source_result.completed,
                                'source_work_out': source_result.work_out,
                                'source_viewed': source_result.viewed,
                                'dest_answers': source_result.answers,
                                'dest_points': source_result.points,
                                'dest_max_points': source_result.max_points,
                                'dest_spent_time': source_result.spent_time,
                                'dest_completed': source_result.completed,
                                'dest_work_out': source_result.work_out,
                                'dest_viewed': source_result.viewed,
                            }

                            result_to_create._log_row = log_row
                            result_to_create._summary = summary
                            results_to_create.append(result_to_create)

                        course_student = (dest_course_lesson.course_id, student_id)
                        course_students.append(course_student)
                        if course_student not in existing_dest_course_students:
                            course_students_to_create.append(course_student)

                    if results_to_update:
                        bulk_update(
                            results_to_update,
                            update_fields=[
                                'answers', 'points', 'max_points', 'spent_time', 'completed', 'work_out', 'viewed',
                            ],
                            batch_size=BULK_BATCH_SIZE_DEFAULT,
                        )

                    if results_to_create:
                        CourseLessonSummary.objects.bulk_create(
                            objs=summaries_to_create,
                            batch_size=BULK_BATCH_SIZE_DEFAULT,
                        )

                        self.stdout.write('Update creating results:')
                        for result_to_create in (tqdm(results_to_create) if progress else results_to_create):
                            result_to_create.summary_id = result_to_create._summary.id

                        CourseLessonResult.objects.bulk_create(
                            objs=results_to_create,
                            batch_size=BULK_BATCH_SIZE_DEFAULT,
                        )

                        self.stdout.write('Create results:')
                        for result_to_create in (tqdm(results_to_create) if progress else results_to_create):
                            log_row = result_to_create._log_row
                            log_row['dest_result_id'] = result_to_create.id
                            log_row['dest_summary_id'] = result_to_create.summary_id
                            csv_writer.writerow(log_row)

                    if course_students_to_create:
                        self.stdout.write('Create course students:')
                        CourseStudent.objects.bulk_create(
                            objs=(
                                CourseStudent(
                                    course_id=course_student[0],
                                    student_id=course_student[1],
                                ) for course_student in course_students_to_create
                            ),
                            batch_size=BULK_BATCH_SIZE_DEFAULT,
                        )

                calculate_lesson_journal.delay(clesson_id=dest_course_lesson_id)
                calculate_course_journal.delay(course_id=dest_course_lesson.course_id)
                recalculate_courselessonstat.delay(clesson_id=dest_course_lesson_id)

                for course_student in course_students:
                    recalculate_studentcoursestat.delay(course_id=course_student[0], student_id=course_student[1])

                if options.get('dry_run', False):
                    raise DryRunException

                self.stdout.write('Commiting...')

        except DryRunException:
            self.stdout.write('Rollback')
        else:
            self.stdout.write('Commit')

    def add_arguments(self, parser):
        parser.add_argument(
            '--dry-run', action='store_true', dest='dry_run',
            help='Run without commit to db',
        )
        parser.add_argument(
            '--progress', action='store_true', dest='progress',
            help='Show progress',
        )
        parser.add_argument(
            '-s', '--source', dest='source_course_lesson_id', required=True,
            help='Source course_lesson_id',
        )
        parser.add_argument(
            '-d', '--dest', dest='dest_course_lesson_id', required=True,
            help='Destination course_lesson_id',
        )
        parser.add_argument(
            '--strategy', dest='conflict_resolve_strategy', required=True, choices=['min', 'max'],
            help='Strategy to resolve conflict on results',
        )
        parser.add_argument(
            '-f', '--file', dest='logfile', required=True,
            help='Logfile (csv) for result movement',
        )

    def handle(self, *args, **options):

        self.move_results(options)

        self.stdout.write("Done\n")
