import logging
from builtins import object, str
from datetime import timedelta

from past.utils import old_div

from django.conf import settings
from django.db import connection, models, transaction
from django.utils import timezone
from django.utils.translation import ugettext_lazy as _

from kelvin.common.fields import JSONField
from kelvin.common.model_mixins import TimeStampMixin
from kelvin.courses.models import CourseLessonLink
from kelvin.lessons.models import Lesson, LessonProblemLink
from kelvin.problems.answers import Answer
from kelvin.results.schemas import student_viewed_problems_validator

logger = logging.getLogger(__name__)


class AbstractLessonSummary(TimeStampMixin, models.Model):
    """
    Общие поля сводки результатов ученика по занятию
    """
    data = JSONField(
        verbose_name=_('Сводка результатов'),
        blank=True,
        null=True,
    )
    student = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        verbose_name=_('Ученик, прошедший занятие'),
        blank=True,
        null=True,
    )
    lesson_finished = models.BooleanField(
        verbose_name=_('Ученик завершил занятие в классе'),
        blank=True,
        default=False,
    )

    class Meta(object):
        abstract = True


class LessonSummary(AbstractLessonSummary):
    """
    Сводка результатов ученика по занятию
    """
    lesson = models.ForeignKey(
        Lesson,
        verbose_name=_('Пройденное занятие'),
    )

    class Meta(object):
        verbose_name = _('Сводка результатов занятия (вне курса)')
        verbose_name_plural = _('Сводки результатов занятий (вне курса)')
        unique_together = ('student', 'lesson')


class CourseLessonSummary(AbstractLessonSummary):
    """
    Сводка результатов ученика по занятию в курсе
    """
    clesson = models.ForeignKey(
        CourseLessonLink,
        verbose_name=_('Пройденное курсозанятие'),
    )

    class Meta(object):
        verbose_name = _('Сводка результатов занятия в курса')
        verbose_name_plural = _('Сводки результатов занятий в курсе')
        unique_together = ('student', 'clesson')


class AbstractLessonResult(TimeStampMixin, models.Model):
    """
    Общие поля попытки пройти занятие
    """
    answers = JSONField(
        verbose_name=_('Ответы на занятие'),
        blank=True,
        null=True,
    )
    points = models.IntegerField(
        verbose_name=_('Количество баллов за занятие'),
        null=True,
        blank=True,
    )
    max_points = models.IntegerField(
        verbose_name=_('Максимальное количество баллов за занятие'),
    )
    spent_time = models.IntegerField(
        verbose_name=_('Количество секунд, потраченное на занятие'),
        blank=True,
        null=True,
    )
    completed = models.BooleanField(
        verbose_name=_('Попытка завершена'),
        default=True,
    )
    work_out = models.BooleanField(
        verbose_name=_('Попытка "вне занятия"'),
        default=False,
    )
    viewed = models.BooleanField(
        verbose_name=_('Пользователь видел результат'),
        default=False,
    )

    class Meta(object):
        abstract = True

    @staticmethod
    def get_summarizable_lessonproblemlinks(lesson_id, link_ids=None):
        """
        Возвращает связи `LessonProblemLink`, которые связывают занятие
        `lesson_id` с обычными задачами и тренажерами, но не теорией. Если
        предоставлен список `link_ids`, то отбирает такие связи только с `id`
        из списка, иначе - все такие связи. Так можно, например, наложить
        ограничение по назначениям ученика в некоторых случаях
        """
        links = LessonProblemLink.objects.filter(
            lesson_id=lesson_id).exclude(type=LessonProblemLink.TYPE_THEORY)
        if link_ids is not None:
            links = links.filter(id__in=link_ids)
        return links

    @staticmethod
    def get_all_lessonproblemlinks_count(lesson_id, link_ids=None):
        """
        Возвращает количество связей `LessonProblemLink`, которые связывают занятие
        `lesson_id` с обычными задачами и тренажерами, в том числе с теорией. Если
        предоставлен список `link_ids`, то отбирает такие связи только с `id`
        из списка, иначе - все такие связи. Так можно, например, наложить
        ограничение по назначениям ученика в некоторых случаях
        """
        links = LessonProblemLink.objects.filter(
            lesson_id=lesson_id)
        if link_ids is not None:
            links = links.filter(id__in=link_ids)
        return links.count()

    @staticmethod
    def get_summary(assigned_problem_links, result, format_as_list=False,
                    lesson_scenario=None, with_max_points=True,
                    force_show_results=False):
        """
        Вернуть сводку результатов

        :param assigned_problem_links: список назначенных связей занятие-вопрос
        :param result: результат ученика по занятию
        :param format_as_list: формат ответа, список (True) или словарь (False)
        :param lesson_scenario: сценарий занятия, по которому оно проводилось
        :param with_max_points: флаг: выгружать ли поле max_points для каждой
                                задачи (будет обращаться в модель Problem)
                                (default True)
        :param force_show_results: показать результаты независимо от
                                   сценария занятия (для учителя)

        Пример ключа `problems`:
        {
            '100': {
                'answered': True,
                'status': 1,
                'time': 100,
                'attempt_number': 4,
                'max_attempts': 5,
            },
            '101': {
                'answered': False,
                'status': None,
            },
        }
        или
        [
            {
                'answered': True,
                'status': 1,
                'time': 100,
                'attempt_number': 4,
                'max_attempts': 5,
            },
            {
                'answered': False,
                'status': None,
            },
        ]
        """
        # TODO: сделать lesson_scenario обязательным аргументом
        # в методе логически смешаны 2 вещи: получение динамической
        # сводки (которая меняется со временем) для ученика и статической
        # сводки (которая может остаться в статистике навечно) для учителя
        if result:
            answers = result.answers
            summary = {
                'points': result.points,
                'max_points': result.max_points,
                'completed': bool(result.completed or
                                  result.time_expired(lesson_scenario)
                                  if lesson_scenario else 0),
            }
        else:
            answers = {}
            summary = {
                'points': 0,
                'max_points': 0,  # FIXME EDU-373
                'completed': False,
            }

        summary['mode'] = lesson_scenario.mode if lesson_scenario else None

        if (
            lesson_scenario and
            lesson_scenario.mode == CourseLessonLink.CONTROL_WORK_MODE
        ):
            if lesson_scenario.evaluation_date and lesson_scenario.evaluation_date > timezone.now():
                hidden_results = True

                # если запрашивают результаты, то считаем контрольную
                # как-будто результаты опубликованы
                is_evaluated_control_work = force_show_results
            else:
                hidden_results = False
                is_evaluated_control_work = True
        elif (lesson_scenario and
                lesson_scenario.mode == CourseLessonLink.DIAGNOSTICS_MODE):
            is_evaluated_control_work = False
            if lesson_scenario.evaluation_date and lesson_scenario.evaluation_date > timezone.now():
                hidden_results = True
            else:
                hidden_results = False
        else:
            hidden_results = False
            is_evaluated_control_work = False

        def get_answer_summary(link):
            """Сводка ответа на задачу"""
            is_evaluable = (lesson_scenario and
                            lesson_scenario.mode in
                            CourseLessonLink.EVALUATION_LESSON_MODES)
            answer_summary = Answer.get_summary(
                answers.get(str(link.id)),
                1 if is_evaluable else link.options.get('max_attempts'),
                hidden_results=hidden_results and not force_show_results,
                force_results=is_evaluated_control_work,
                max_points=(link.options.get('max_points') or
                            link.problem.max_points),
            )
            answer_summary['type'] = LessonProblemLink.TYPE_COMMON

            # TODO надо посмотреть на наличие select_related-ов
            if with_max_points:
                answer_summary['max_points'] = (
                    link.options.get('max_points') or
                    link.problem.max_points
                )
            return answer_summary

        if format_as_list:
            summary['problems'] = [
                get_answer_summary(problem_link)
                for problem_link in assigned_problem_links
            ]
        else:
            summary['problems'] = {
                problem_link.id: get_answer_summary(problem_link)
                for problem_link in assigned_problem_links
            }
        return summary

    def get_count(self, status, assigned_problem_links,
                  summary=None, answered=True):
        """
        Число ответов в попытке по какому-нибудь статусу
        """
        summary = summary or self.get_summary(assigned_problem_links, self)
        return sum(
            1
            for problem_id, problem_summary in summary['problems'].items()
            if (problem_summary['status'] == status and
                problem_summary['answered'] == answered)
        )

    def get_incorrect_count(self, assigned_problem_links, summary=None):
        """
        Число неправильных ответов в попытке
        """
        return self.get_count(
            Answer.SUMMARY_INCORRECT, assigned_problem_links, summary
        )

    def get_correct_count(self, assigned_problem_links, summary=None):
        """
        Число правильных ответов в попытке
        """
        return self.get_count(
            Answer.SUMMARY_CORRECT, assigned_problem_links, summary
        )

    def get_unanswered_count(self, assigned_problem_links, summary=None):
        """
        Число вопросов, на которые нет ответа
        """
        return self.get_count(
            None, assigned_problem_links, summary, answered=False
        )

    def get_all_files(self):
        """
        Возвращает массив файлов для текущего результата (из всех попыток по
        всем задачам)
        """
        files = []
        for answer in self.answers:
            for attempt in self.answers[answer]:
                if isinstance(attempt.get('custom_answer'), list):
                    for custom_answer in attempt['custom_answer']:
                        files += custom_answer.get('files', [])
        return files

    def time_expired(self, scenario):
        """
        Говорит, что попытка истекла согласно указанному сценарию занятия.
        """
        return (scenario.duration and scenario.duration <=
                old_div((timezone.now() - self.date_created).total_seconds(), 60))

    def quiz_time_limit(self, quiz_scenario):
        """
        Возвращает дату-время последнего возможного ответа на контрольную
        (или диагностику)
        """
        return min(
            self.date_created + timedelta(
                minutes=quiz_scenario.duration),
            quiz_scenario.finish_date,
        )


class LessonResult(AbstractLessonResult):
    """
    Результаты на попытку прохождения занятия (вне курсе)
    """
    summary = models.ForeignKey(
        LessonSummary,
        verbose_name=_('Сводка результатов занятия'),
    )

    class Meta(object):
        verbose_name = _('Результат занятия (вне курса)')
        verbose_name_plural = _('Результаты занятий (вне курса)')


class CourseLessonResultMeta(TimeStampMixin, models.Model):
    """
    Метаинформация о прохождении занятия в курсе
    """
    student_viewed_problems = JSONField(
        verbose_name=_('Задачи, где ученик видел ответ учителя'),
        validators=[student_viewed_problems_validator],
        default=dict({}),
        blank=True,
    )

    class Meta(object):
        verbose_name = _('Метаинформация о курсозанятии')
        verbose_name_plural = _('Метаинформация о курсозанятиях')


class CourseLessonResult(AbstractLessonResult):
    """
    Результаты на попытку прохождения занятия в курсе
    """
    summary = models.ForeignKey(
        CourseLessonSummary,
        verbose_name=_('Сводка результатов занятия'),
    )
    meta = models.ForeignKey(
        CourseLessonResultMeta,
        verbose_name=_('Метаинформация о занятии в курсе'),
        null=True,
        blank=True,
    )

    class Meta(object):
        verbose_name = _('Результат занятия в курсе')
        verbose_name_plural = _('Результаты занятий в курсе')

        permissions = [
            ("backup_task_status", "Can backup CourseLessonResult"),
        ]

    def get_problem_stats(self):
        result = {}

        # проблемлинки нам нужны, чтобы получить порядок их следования в модуле
        # этот порядок может быть важен для получателя результатов данного метода
        problem_order = {}
        for item in self.summary.clesson.lesson.lessonproblemlink_set.all():
            problem_order[item.id] = item.order

        for problem_link_id, problem_answers in list(self.answers.items()):
            problem_link_id = int(problem_link_id)

            if problem_answers:
                item = problem_answers[-1]  # засчитываем только последнюю попытку в независимости от ее правильности

                if item["completed"]:
                    result[problem_link_id] = {
                        "id": problem_link_id,
                        "result": "fail" if item["mistakes"] > 0 else "ok",
                        "order": problem_order.get(problem_link_id, 0),
                    }

                    # безопасно удаляем обработанные проблемы
                    # а те, что останутся будем считать неотвеченными
                    problem_order.pop(problem_link_id, None)

        for unanswered_problem_link_id in problem_order:
            result[unanswered_problem_link_id] = {
                "id": unanswered_problem_link_id,
                "result": "noanswer",
                "order": problem_order[unanswered_problem_link_id]
            }

        return result


class CourseLessonResultHook(models.Model):
    course_lesson_result = models.ForeignKey(
        to="results.CourseLessonResult",
        verbose_name=_("Результат занятия в курсе"),
    )
    hook_info = JSONField(
        verbose_name=_('Хук на состояния модуля'),
        blank=True,
        default={},
        help_text=_('Настройки хуков для интеграции с внешними системами (например Я.Лицей)')
    )
    send_time = models.DateTimeField(
        verbose_name=_('Время отправки хука'),
        null=True,
        blank=True,
    )

    def __str__(self):
        hooks = list(self.hook_info.keys()) if isinstance(self.hook_info, dict) else []
        return f'result: {self.course_lesson_result}, hooks: {hooks}'


@transaction.atomic
def backup_clesson_results(clesson_result_ids):
    if not clesson_result_ids:
        return

    clesson_result_ids = tuple(clesson_result_ids)
    with connection.cursor() as cursor:
        cursor.execute("INSERT INTO results_courselessonresult_backup SELECT * from results_courselessonresult "
                       "WHERE id IN %s", (clesson_result_ids, ))
        cursor.execute("DELETE FROM results_courselessonresult WHERE id IN %s", (clesson_result_ids, ))
