from builtins import map, object, range
from datetime import timedelta

from django.conf import settings
from django.utils import timezone

from rest_framework import fields, serializers

from kelvin.accounts.serializers import BaseUserSerializer
from kelvin.common.serializer_fields import CurrentUserOrNoneDefault
from kelvin.common.serializer_mixins import PreciseDateUpdatedFieldMixin, SerializerForeignKeyMixin
from kelvin.courses.models import CourseLessonLink
from kelvin.lesson_assignments.models import LessonAssignment
from kelvin.lessons.models import Lesson, LessonScenario
from kelvin.lessons.serializer_fields import AnswersField
from kelvin.problems.answers import CustomAnswer
from kelvin.problems.serializers import AnswerDatabaseSerializer, AnswerSerializer
from kelvin.problems.utils import get_markers_dict_from_layout
from kelvin.results.models import CourseLessonResult, CourseLessonSummary, LessonResult, LessonSummary


class LessonSummarySerializer(serializers.ModelSerializer):
    """
    Сериализатор сводки результатов занятия
    """
    class Meta(object):
        model = LessonSummary
        fields = '__all__'


class CourseLessonSummarySerializer(serializers.ModelSerializer):
    """
    Сериализатор сводки результатов занятия в курсе
    """
    class Meta(object):
        model = CourseLessonSummary
        fields = '__all__'


class LessonSummaryInResultSerializer(serializers.ModelSerializer):
    """
    Сериализатор для создания сводки при создании результатов
    """
    student = serializers.PrimaryKeyRelatedField(
        required=False,
        read_only=True,
        default=serializers.CurrentUserDefault(),
    )

    class Meta(object):
        model = LessonSummary
        fields = (
            'lesson',
            'student',
        )

    def create(self, validated_data):
        """
        Создаем объект только при его отсутствии
        """
        if validated_data['student']:
            return LessonSummary.objects.get_or_create(
                lesson=validated_data['lesson'],
                student=validated_data['student'],
            )[0]
        else:
            return LessonSummary.objects.create(
                lesson=validated_data['lesson'],
                student=None,
            )

    def update(self, instance, validated_data):
        """
        Не обновляем объект
        """
        return instance


class CourseLessonSummaryInResultSerializer(serializers.ModelSerializer):
    """
    Сериализатор для создания сводки при создании результатов
    """
    student = serializers.PrimaryKeyRelatedField(
        required=False,
        read_only=True,
        default=CurrentUserOrNoneDefault(),
    )

    class Meta(object):
        model = CourseLessonSummary
        fields = (
            'clesson',
            'student',
        )

    def create(self, validated_data):
        """
        Создаем объект только при его отсутствии для существующего
        пользователя, для анонимного - всегда создаем новый объект
        """
        if validated_data['student']:
            return CourseLessonSummary.objects.get_or_create(
                clesson=validated_data['clesson'],
                student=validated_data['student'],
            )[0]
        else:
            return CourseLessonSummary.objects.create(
                clesson=validated_data['clesson'],
                student=None,
            )

    def update(self, instance, validated_data):
        """
        Не обновляем объект
        """
        return instance


class LessonResultSerializer(PreciseDateUpdatedFieldMixin,
                             SerializerForeignKeyMixin,
                             serializers.ModelSerializer):
    """
    Сериализатор результата прохождения занятия
    """
    answers = AnswersField(required=False, default={})

    # убираем валидацию уникальности занятия и ученика
    summary = LessonSummaryInResultSerializer(validators=tuple())

    fk_update_fields = ['summary']

    default_error_messages = {
        'completed instance': 'can\'t modify completed result',
        'uncompleted attempt': 'uncompleted attempt can be only last',
        'attempt limit': 'attempt limit exceeded in problems {problem_ids}',
        'incomplete attempt limit': 'there can be only one incomplete attempt',
        'already has correct answer':
            'should be no answer after correct one in problems {problem_ids}'
    }

    class Meta(object):
        model = LessonResult
        extra_kwargs = {
            'points': {'read_only': True},
            'max_points': {'read_only': True},
            'points': {'read_only': True},
            'max_points': {'read_only': True},
        }
        fields = (
            'id',
            'date_created',
            'date_updated',
            'answers',
            'points',
            'max_points',
            'spent_time',
            'completed',
            'work_out',
            'summary',
            'date_created',
            'date_updated',
            'viewed',
        )

    def _get_lesson(self, attrs):
        """
        Вернуть занятие для проверки ответов при валидации

        :param attrs: валидируемые данные
        """
        if 'summary' in attrs:
            if isinstance(attrs['summary'], LessonSummary):
                return attrs['summary'].lesson
            else:
                lesson = attrs['summary'].get('lesson')
                # FIXME подумать, как переписать (Youtrack: EDU-171)
                if lesson is not None:
                    if not isinstance(lesson, Lesson):
                        lesson = Lesson.objects.get(id=lesson)
                    return lesson
        return self.instance.summary.lesson

    def validate(self, attrs):
        """
        Проверяет полученные ответы, а также то, что не создается еще одна
        незавершенная попытка на этот урок этим пользователем
        """
        if self.instance:
            if (
                self.instance.completed and not (
                    'update_only_custom_answer' in self.context and
                    self.context['update_only_custom_answer']['type'] in
                    CustomAnswer.Type.TYPES_THAT_MAY_BE_IN_COMPLETED_ATTEMPTS
                )
            ):
                self.fail('completed instance')
        elif (not self.instance and not attrs.get('completed') and
              LessonResult.objects.filter(
                  summary__student=attrs.get('summary', {}).get('student'),
                  summary__lesson=self._get_lesson(attrs),
                  completed=False).exists()):
            self.fail('incomplete attempt limit')

        attrs = super(LessonResultSerializer, self).validate(attrs)

        if 'answers' not in attrs:
            # позволяем менять атрибуты результата отдельно от ответов
            return attrs

        # обрабатываем только ответы на обычные вопросы
        lesson = self._get_lesson(attrs)
        problem_links_dict = lesson.problem_links_dict
        theory_links_dict = lesson.theory_links_dict

        # десериализуем ответы
        answers, theory_answers = {}, {}
        for problem_link_id, user_answer in attrs['answers'].items():
            if problem_link_id in problem_links_dict:
                problem = problem_links_dict[problem_link_id].problem
                if not isinstance(user_answer, list):
                    # если не список, то прислали только одну попытку
                    user_answer = [user_answer]
                serializer = AnswerSerializer(
                    data=user_answer,
                    context={
                        'markers': get_markers_dict_from_layout(
                            problem.markup['layout']),
                        'problem': problem,
                        'request': self.context.get('request', {}),
                    },
                    many=True,
                )
                serializer.is_valid(raise_exception=True)
                answer_objects = serializer.save()

                # может быть только одна последняя незаконченная попытка
                for i, attempt in enumerate(answer_objects):
                    if (not attempt.completed and
                            i != len(answer_objects) - 1):
                        self.fail('uncompleted attempt')
                answers[problem_link_id] = answer_objects
            elif problem_link_id in theory_links_dict:
                if not isinstance(user_answer, list):
                    # если не список, то прислали только одну попытку
                    user_answer = [user_answer]
                serializer = AnswerSerializer(
                    data=user_answer,
                    context=dict(request=self.context.get('request', {})),
                    many=True,
                )
                serializer.is_valid(raise_exception=True)
                answer_objects = serializer.save()
                theory_answers[problem_link_id] = answer_objects

        # проверка ответов
        checked_answers = Lesson.check_answers(
            {
                link_id: link.problem
                for link_id, link in problem_links_dict.items()
            },
            answers,
        )

        # сериализация проверенных ответов
        serializer = AnswerDatabaseSerializer(many=True)
        attrs['answers'] = {
            problem_id: serializer.to_representation(answer)
            for problem_id, answer in checked_answers.items()
        }
        attrs = self._validate_answers(attrs, problem_links_dict)

        theory_answers = {
            link_id: serializer.to_representation(answers)
            for link_id, answers in theory_answers.items()
        }
        attrs['answers'].update(self._validate_theory_answers(theory_answers, theory_links_dict))

        return attrs

    def _validate_answers(self, validated_data, problem_links_dict):
        """
        Проверка, что не превышен лимит попыток ответа на вопросы
        и ранее не было правильного ответа

        Если последняя попытка на вопрос неокончена, то ее перезаписывает,
        в остальных случаях дописывает пришедшие попытки
        """
        if 'answers' not in validated_data:
            return validated_data

        if self.instance and self.partial:
            # изменяем ответы в `validated_data`, учитывая предыдущие ответы
            validated_answers = {}
            for problem_link_id in problem_links_dict.keys():
                attempts = validated_data['answers'].get(problem_link_id, [])
                db_attempts = self.instance.answers.get(problem_link_id, [])
                if not (attempts or db_attempts):
                    # если нет попыток, пропускаем вопрос
                    continue
                if not attempts:
                    # оставляем старые попытки, если нет ничего нового
                    validated_answers[problem_link_id] = db_attempts
                elif (hasattr(self, 'initial_data') and isinstance(
                        self.initial_data['answers'][problem_link_id], list)):
                    # обновляем весь список попыток, если пришел список попыток
                    validated_answers[problem_link_id] = attempts
                elif db_attempts and not db_attempts[-1]['completed']:
                    # перезаписываем последнюю попытку, если она не завершена
                    validated_answers[problem_link_id] = (
                        db_attempts[:-1] + attempts)
                else:
                    validated_answers[problem_link_id] = db_attempts + attempts

            validated_data['answers'] = validated_answers

        # проверяем, что по каждой задаче отсутствует правильный ответ,
        # который не стоит последним
        problems_already_true = []
        for problem_link_id, answers in validated_data['answers'].items():
            count_answers = len(answers)
            for i, answer in enumerate(answers):
                if len(answer.get('custom_answer', []) or []):
                    break
                if answer['mistakes'] == 0 and i != count_answers - 1:
                    problems_already_true.append(problem_link_id)
                    break
        if problems_already_true:
            self.fail('already has correct answer',
                      problem_ids=problems_already_true)

        # проверяем лимит попыток
        attempt_limit_problems = []
        for problem_link_id, answers in validated_data['answers'].items():
            options = problem_links_dict[problem_link_id].options
            # FIXME EDU-557 всегда должны быть `max_attempts`
            if options and options['max_attempts'] < len(answers):
                attempt_limit_problems.append(problem_link_id)
        if attempt_limit_problems:
            self.fail('attempt limit', problem_ids=attempt_limit_problems)

        return validated_data

    def _validate_theory_answers(self, theory_answers, theory_links_dict):

        if self.instance and self.partial:
            # изменяем ответы в `theory_answers`, учитывая предыдущие ответы
            validated_answers = {}
            for theory_link_id in theory_links_dict.keys():
                attempts = theory_answers.get(theory_link_id, [])
                db_attempts = self.instance.answers.get(theory_link_id, [])
                if not attempts and not db_attempts:
                    # если нет попыток, пропускаем вопрос
                    continue
                if not attempts:
                    # оставляем старые попытки, если нет ничего нового
                    validated_answers[theory_link_id] = db_attempts
                elif (hasattr(self, 'initial_data') and isinstance(
                        self.initial_data['answers'][theory_link_id], list)):
                    # обновляем весь список попыток, если пришел список попыток
                    validated_answers[theory_link_id] = attempts
                elif db_attempts and not db_attempts[-1]['completed']:
                    # перезаписываем последнюю попытку, если она не завершена
                    validated_answers[theory_link_id] = (
                        db_attempts[:-1] + attempts)
                else:
                    validated_answers[theory_link_id] = db_attempts + attempts

            theory_answers = validated_answers

        return theory_answers

    def create(self, validated_data):
        """
        Считает число баллов ученика
        """
        lesson = self._get_lesson(validated_data)
        validated_data['points'] = lesson.get_points(validated_data.get('answers', {}))
        validated_data['max_points'] = lesson.get_max_points()
        return super(LessonResultSerializer, self).create(validated_data)

    def update(self, lesson_result, validated_data):
        """
        Считает число баллов ученика
        """
        if 'answers' in validated_data:
            lesson = self._get_lesson(validated_data)
            validated_data['points'] = lesson.get_points(validated_data['answers'])
            validated_data['max_points'] = lesson.get_max_points()

            # С фронта приходят только данные, которые необходимо заменить.
            # В стандартном сериализаторе `spent_time` проставляется в None и
            # перезатирает старое значение. Данный код вручную копирует поле
            # `spent_time`
            # TODO: Найти более красивое решение
            for link_id in list(validated_data['answers'].keys()):
                for attempt in range(len(validated_data['answers'][link_id])):
                    if (
                        (len(lesson_result.answers.get(link_id, [])) >
                            attempt) and
                        validated_data['answers'][link_id][attempt].get(
                            'spent_time') is None
                    ):
                        validated_data['answers'][link_id][attempt][
                            'spent_time'] = (
                                lesson_result.answers[link_id][attempt].get(
                                    'spent_time')
                        )

        return super(LessonResultSerializer, self).update(
            lesson_result, validated_data)

    def to_representation(self, instance):
        """
        Если надо скрыть ответы, убираем баллы за занятие
        """
        data = super(LessonResultSerializer, self).to_representation(instance)

        if self.context.get('hide_answers'):
            data.pop('points')
        summary = data.pop('summary')
        data['student'] = summary['student']
        data['lesson'] = summary['lesson']

        return data

    def to_internal_value(self, data):
        """
        Добавляет сводку результатов, поддержка старого формата ответов
        """
        if 'lesson' in data:
            data['summary'] = {'lesson': data.pop('lesson')}

        return super(LessonResultSerializer, self).to_internal_value(data)


class CourseLessonResultSerializer(LessonResultSerializer):
    """
    Сериализатор результата прохождения занятия в курсе
    """
    # убираем валидацию уникальности занятия и ученика
    summary = CourseLessonSummaryInResultSerializer(validators=tuple())

    fk_update_fields = ['summary']

    student_viewed_problems = serializers.SerializerMethodField()

    def get_student_viewed_problems(self, attrs):
        meta_obj = attrs.meta
        return meta_obj.student_viewed_problems if meta_obj is not None else {}

    default_error_messages = {
        'attempts limit': 'student reaches attempt limit',
        'lesson completed': 'clesson is completed',
        'incomplete attempt limit': 'there can be only one incomplete attempt',
        'attempt only workout': 'can not modify to non-workout attempt',
        'time limit': 'time limit exceeded',
        'future date assignment': 'can not create result for future clesson',
        'course not for anonymous': 'authenticate to create result',
        'not enough time': 'not enough time to complete clesson',
    }

    class Meta(object):
        model = CourseLessonResult
        extra_kwargs = {
            'points': {'read_only': True},
            'max_points': {'read_only': True},
            'work_out': {'default': False},
        }
        fields = (
            'id',
            'date_created',
            'date_updated',
            'answers',
            'points',
            'max_points',
            'spent_time',
            'completed',
            'work_out',
            'summary',
            'viewed',
            'student_viewed_problems',
        )

    def _get_clesson(self, attrs):
        """
        Вернуть курсозанятие

        :returns CourseLessonLink:
        """
        if 'summary' in attrs:
            if isinstance(attrs['summary'], CourseLessonSummary):
                return attrs['summary'].clesson
            else:
                clesson = attrs['summary'].get('clesson')
                # FIXME подумать, как переписать (Youtrack: EDU-171)
                if clesson is not None:
                    if not isinstance(clesson, CourseLessonLink):
                        clesson = CourseLessonLink.objects.get(id=clesson)
                    return clesson
        return self.instance.summary.clesson

    def _get_lesson(self, attrs):
        """
        Вернуть занятие для проверки ответов при валидации

        :param attrs: валидируемые данные
        """
        return self._get_clesson(attrs).lesson

    def _get_assigned_problem_ids(self, validated_data):
        """
        Список идентификаторов назначенных ученику задач
        """
        # TODO write memoize
        if not hasattr(self, '_assigned_problem_ids'):
            if self.context['request'].user.is_authenticated:
                self._assigned_problem_ids = (
                    LessonAssignment.get_student_problems(
                        self.context['request'].user,
                        self._get_clesson(validated_data),
                    )
                )
            else:
                self._assigned_problem_ids = None
        return self._assigned_problem_ids

    def validate(self, attrs):
        """
        # TODO: разбить валидацию на отдельные методы,
        вынести в отдельный класс

        Проверяет, что:

        * если ученик только начинает проходить контрольную или диагностику
        и до конца осталось меньше времени, чем продолжительность
        контрольной, то результаты не принимаются (EDUCATION-595)
        * анонимный пользователь создает попытку только в курсе для анонимного
        пользователя,
        * в завершенное занятие записывается внеурочная попытка,
        * не истекло число попыток при создании,
        * занятие для учеников назначено на прошедшее время,
        * создается только одна незавершенная попытка (по одной в классе и
        вне класса).

        Если курсозаниятие является контрольной или диагностикой, также
        проверяет истечение лимита времени на отправку ответа.
        Отбрасывает неназначенные задачи
        """
        clesson = self._get_clesson(attrs)
        student = (
            self.instance.summary.student if self.instance else
            attrs.get('summary', {}).get('student')
        )
        update_only_custom_answer = self.context.get('update_only_custom_answer')

        if not update_only_custom_answer:
            self._validate_if_results_accepted(clesson)

        # анонимный пользователь может создавать попытку только в
        # соответствующем курсе
        if not (self.instance or student or clesson.course.allow_anonymous):
            self.fail('course not for anonymous')

        # может создаваться только одна незавершенная попытка для
        # попытки неанонимного пользователя
        if (not self.instance and student and not attrs.get('completed') and
                CourseLessonResult.objects.filter(
                summary__student=student,
                summary__clesson=clesson,
                completed=False,
                work_out=attrs.get('work_out')).exists()):
            self.fail('incomplete attempt limit')

        # для учеников курсозанятие должно быть назначено на прошедшее время
        if ((not clesson.date_assignment or
                clesson.date_assignment > timezone.now()) and
                not (student and (
                    student.is_teacher or student.is_content_manager))):
            self.fail('future date assignment')

        # Для контрольных и диагностик проверяем, не истекло ли время отправки
        # результата и при этом это не обновление сообщения ручной проверки,
        # которое можно отправлять после завершения контрольной
        types_after_control_work = CustomAnswer.Type.TYPES_THAT_MAY_BY_AFTER_CONTROL_WORK
        if (
            clesson.mode in CourseLessonLink.EVALUATION_LESSON_MODES and not
            (update_only_custom_answer and update_only_custom_answer.get('type') in types_after_control_work)
        ):
            time_limit = (
                self.instance.quiz_time_limit(clesson) if self.instance
                else clesson.finish_date
            )
            if timezone.now() >= time_limit:
                self.fail('time limit')

        if (
            self.instance and
            self.instance.work_out is True and
            attrs.get('work_out') is False
        ):
            # нельзя изменить "внеурочную" попытку на "урочную"
            self.fail('attempt only workout')

        if (
            clesson.date_completed and clesson.date_completed < timezone.now() and not (
                attrs.get('work_out') or
                self.instance and
                self.instance.work_out
            )
        ):
            # в завершенном занятии разрешаем создавать или изменять
            # только "внеурочное" занятие
            self.fail('lesson completed')

        if clesson.finish_date and (
            clesson.finish_date < timezone.now() and
            not (
                attrs.get('work_out') or
                self.instance and self.instance.work_out
            )
        ):
            self.fail('lesson completed')

        if student and not (self.instance or attrs.get('work_out')) and not update_only_custom_answer:
            # число попыток проверяем только при создании "урочного" занятия
            # и для неанонимной попытки
            # и не проверяем попытки при обновлении комментария
            attempts_count = CourseLessonResult.objects.filter(
                summary__student=attrs.get('summary', {}).get('student'),
                summary__clesson=clesson,
                work_out=False,
            ).count()
            if attempts_count >= clesson.max_attempts_in_group:
                self.fail('attempts limit')

        validated_data = super(CourseLessonResultSerializer, self).validate(
            attrs)

        if 'answers' not in validated_data:
            return validated_data

        # оставляем только назначенные задачи, иначе баллы будут больше
        # максимума
        assigned = self._get_assigned_problem_ids(validated_data)
        if assigned is not None:
            assigned = set(map(str, assigned))
            for extra in (set(validated_data['answers'].keys()) - assigned):
                validated_data['answers'].pop(extra)

        return validated_data

    def _validate_if_results_accepted(self, clesson):
        """
        Если недостаточно времени, чтобы пройти контрольную – останавливаем
        прием результатов, это должен быть первый запрос на создание
        результата.
        """
        if not self.partial and not clesson.can_accept_results:
            self.fail('not enough time')

    def create(self, validated_data):
        """
        Считает число баллов ученика, учитывая назначение
        """
        lesson = self._get_lesson(validated_data)
        validated_data['max_points'] = lesson.get_max_points(self._get_assigned_problem_ids(validated_data))
        validated_data['points'] = lesson.get_points(validated_data.get('answers', {}))
        # вызываем `ModelSerializer.create`
        return super(LessonResultSerializer, self).create(validated_data)

    def to_representation(self, instance):
        """
        Если надо скрыть ответы, убираем баллы за занятие
        """
        # вызываем не родительский метод
        data = super(LessonResultSerializer, self).to_representation(instance)

        if self.context.get('hide_answers'):
            data.pop('points')
        summary = data.pop('summary')
        data['student'] = summary['student']
        data['clesson'] = summary['clesson']

        clesson_result = CourseLessonResult.objects.get(pk=data['id'])
        if (clesson_result.summary.clesson.mode in
                LessonScenario.EVALUATION_LESSON_MODES):
            duration = clesson_result.summary.clesson.duration
            if duration:
                data['completed'] = data['completed'] or (
                    clesson_result.date_created + timedelta(minutes=duration) <
                    timezone.now()
                )

        return data

    def to_internal_value(self, data):
        """
        Добавляет сводку результатов, поддержка старого формата ответов
        """
        if 'clesson' in data:
            data['summary'] = {'clesson': data.pop('clesson')}

        # вызываем не родительский метод
        return super(LessonResultSerializer, self).to_internal_value(data)


class LessonResultInfoInEventSerializer(serializers.ModelSerializer):
    """
    Сериализатор результата занятия мероприятия.
    Может вызываться, когда занятия еще нет.
    """
    created = fields.SerializerMethodField()
    completed = fields.SerializerMethodField()
    evaluated = fields.SerializerMethodField()

    def get_created(self, obj):
        """
        У пользователя есть попытка по занятию
        """
        return obj.id is not None

    def get_completed(self, obj):
        """
        Попытка пользователя завершена или вышло время выполнения задачи
        """
        if obj.id and obj.completed:
            return True

        scenario = self.context.get('scenario', {})
        if scenario:
            return obj.time_expired(LessonScenario(**scenario))

        return False

    def get_evaluated(self, obj):
        """
        Попытка пользователя проверена
        Возвращает `True` для тренировочных занятий, а также для основных
        занятий, если наступила дата публикации и ответы были проверены, т.е.
        есть очки
        """
        lesson_type = self.context.get('lesson_type')
        if lesson_type == 'training_lesson':
            return True
        publication_date = self.context.get('publication_date')
        if publication_date:
            return (
                timezone.now() >= publication_date and
                obj.points is not None
            )
        return False

    class Meta(object):
        model = LessonResult
        fields = (
            'created',
            'completed',
            'evaluated',
        )


class LessonResultInEventSerializer(serializers.ModelSerializer):
    """
    Сериализатор результата занятия мероприятия
    """
    student = BaseUserSerializer(source='summary.student')
    max_points = fields.SerializerMethodField()
    unchecked_problems = fields.SerializerMethodField()

    class Meta(object):
        model = LessonResult
        fields = (
            'student',
            'points',
            'max_points',
            'unchecked_problems',
        )

    def get_max_points(self, obj):
        """
        Максимально возможное кол-во баллов в занятии
        """
        return self.context.get('max_points')

    def get_unchecked_problems(self, obj):
        """
        Кол-во непроверенных задач в попытке
        """
        return len([answer for answer in list(obj.answers.values()) if answer[0]['mistakes'] is None])


class ResetClessonResultsInputSerializer(serializers.Serializer):
    course_id = serializers.IntegerField(required=True)
    username = serializers.CharField(required=True)
    clesson_id = serializers.CharField(required=False)

    def validate_course_id(self, value):
        if value not in settings.RESETTABLE_COURSES_IDS:
            raise serializers.ValidationError(
                "Course can not be reset"
            )
        return value


class ResetClessonResultsInputSerializer3(serializers.Serializer):
    course_id = serializers.IntegerField(required=True)
    username = serializers.CharField(required=True)
    clesson_id = serializers.IntegerField(required=False)
