import logging

import requests

from django.conf import settings
from django.utils import timezone

from kelvin.common.tvm import get_service_ticket

logger = logging.getLogger(__name__)


class SendHookException(Exception):
    pass


def send_start_hook(course_lesson_result_hook, start_hook_info, commit=True):
    logger.debug(
        "Processing hook '{}' of the lesson {}".format(
            "start", course_lesson_result_hook.course_lesson_result.summary.clesson_id,
        )
    )
    url = start_hook_info['url']
    method = start_hook_info.get('method', 'get').lower()
    payload = {
        "action": "start",
        "username": course_lesson_result_hook.course_lesson_result.summary.student.username,
        "yauid": course_lesson_result_hook.course_lesson_result.summary.student.yauid,
        "course_id": course_lesson_result_hook.course_lesson_result.summary.clesson.course_id,
        "module_id": course_lesson_result_hook.course_lesson_result.summary.clesson.id,
        "datetime": course_lesson_result_hook.course_lesson_result.date_created.isoformat(),
    }
    params = {}
    data = {}
    if method == 'get':
        params = payload
    else:
        data = payload

    response = requests.request(
        method=method,
        url=url,
        params=params,
        json=data,
        verify=False,
        headers={
            "X-Ya-Service-Ticket": get_service_ticket(settings.LYCEUM_TVM_CLIENT_ID),
        },
        cookies={
        }
    )
    if response.ok:
        logger.info('Start hook sent: OK')
        course_lesson_result_hook.send_time = timezone.now()
        if commit:
            course_lesson_result_hook.save()
    else:
        logger.error(f'Got {response.status_code} response code from lyceum.')
        error_message = f'Start hook sent: FAILED. Status: {response.status_code}. Content: {response.content}'
        logger.error(error_message)
        raise SendHookException(error_message)


def send_stop_hook(course_lesson_result_hook, stop_hook_info, commit=True):
    logger.debug(
        "Processing hook '{}' of the lesson {}".format(
            "stop", course_lesson_result_hook.course_lesson_result.summary.clesson_id,
        )
    )
    url = stop_hook_info['url']
    method = stop_hook_info.get('method', 'get').lower()
    payload = {
        "action": "stop",
        "username": course_lesson_result_hook.course_lesson_result.summary.student.username,
        "yauid": course_lesson_result_hook.course_lesson_result.summary.student.yauid,
        "course_id": course_lesson_result_hook.course_lesson_result.summary.clesson.course_id,
        "module_id": course_lesson_result_hook.course_lesson_result.summary.clesson.id,
        "datetime": course_lesson_result_hook.course_lesson_result.date_created.isoformat(),
        "tests": course_lesson_result_hook.course_lesson_result.get_problem_stats(),
    }
    params = {}
    data = {}
    if method == 'get':
        params = payload
    else:
        data = payload
    response = requests.request(
        method=method,
        url=url,
        params=params,
        json=data,
        verify=False,
        headers={
            "X-Ya-Service-Ticket": get_service_ticket(settings.LYCEUM_TVM_CLIENT_ID),
        },
        cookies={
        }
    )
    if response.ok:
        logger.info('Stop hook sent: OK')
        course_lesson_result_hook.send_time = timezone.now()
        if commit:
            course_lesson_result_hook.save()
    else:
        logger.error(f'Got {response.status_code} response code from lyceum.')
        error_message = f'Stop hook sent: FAILED. Status: {response.status_code}. Content: {response.content}'
        logger.error(error_message)
        raise SendHookException(error_message)


def send_hook(course_lesson_result_hook, commit=True):
    if not isinstance(course_lesson_result_hook.hook_info, dict):
        logger.error('hook_info is not a dict, CourseLessonResultHook({})'.format(course_lesson_result_hook.id))
        return

    if 'start' in course_lesson_result_hook.hook_info:
        start_hook_info = course_lesson_result_hook.hook_info['start']
        send_start_hook(
            course_lesson_result_hook=course_lesson_result_hook,
            start_hook_info=start_hook_info,
            commit=commit,
        )

    elif 'stop' in course_lesson_result_hook.hook_info:
        stop_hook_info = course_lesson_result_hook.hook_info['stop']
        send_stop_hook(
            course_lesson_result_hook=course_lesson_result_hook,
            stop_hook_info=stop_hook_info,
            commit=commit,
        )
