from logging import getLogger

from django.db.models.signals import post_save, pre_save

from kelvin.accounts.models import User
from kelvin.courses.models import Criterion
from kelvin.results.models import CourseLessonResult, CourseLessonResultHook
from kelvin.results.tasks import pull_triggers_task, send_hook_task

logger = getLogger(__name__)
sentry_logger = getLogger('sentry')


def process_lesson_result_start_hooks(sender, instance, **kwargs):
    """
    Обрабатываем start-хуки модуля, если они есть
    """
    hooks = instance.summary.clesson.lesson.hooks
    if hooks and instance.summary.student:
        start_hook_info = hooks.get("start", {})
        if start_hook_info:
            obj, created = instance.courselessonresulthook_set.filter(
                hook_info__has_key="start",
            ).get_or_create(
                defaults={
                    "course_lesson_result_id": instance.id,
                    "hook_info": {"start": start_hook_info},
                },
            )
            if created:
                logger.debug("Creating hook '{}' of the lesson {}".format("start", instance.summary.clesson_id))


def process_lesson_result_stop_hooks(sender, instance, **kwargs):
    """
    Обрабатываем stop-хуки модуля, если они есть
    """
    hooks = instance.summary.clesson.lesson.hooks
    if hooks and instance.summary.student:
        stop_hook_info = hooks.get("stop", {})
        if stop_hook_info and instance.completed:
            # засекли момент завершения модуля - надо создать stop-хук
            obj, created = instance.courselessonresulthook_set.filter(
                hook_info__has_key="stop"
            ).get_or_create(
                defaults={
                    "course_lesson_result_id": instance.id,
                    "hook_info": {"stop": stop_hook_info},
                },
            )
            if created:
                logger.debug("Creating hook '{}' of the lesson {}".format("stop", instance.summary.clesson_id))


def process_lesson_results_data(sender, instance, **kwargs):
    if instance.pk and instance.summary.student:
        old_state = CourseLessonResult.objects.get(pk=instance.pk)
        if old_state.answers != {} and instance.answers == {}:
            raise RuntimeError("Attempt to zero answer detected")


def pull_triggers(sender, instance, **kwargs):
    pull_triggers_task.delay(instance.id)


def send_hook_signal(sender, instance, **kwargs):
    if not instance.send_time:
        send_hook_task.delay(instance.id)

#  use post- because we need the 'created' kwarg
post_save.connect(pull_triggers, CourseLessonResult)

#  use pre- because we need the old state
pre_save.connect(process_lesson_results_data, CourseLessonResult)

post_save.connect(process_lesson_result_start_hooks, CourseLessonResult)
post_save.connect(process_lesson_result_stop_hooks, CourseLessonResult)
post_save.connect(send_hook_signal, CourseLessonResultHook)
