from celery.utils.log import get_task_logger

from django.conf import settings
from django.db import transaction

from kelvin.celery import app
from kelvin.common.decorators.task_logger import logged_task
from kelvin.results.models import CourseLessonResult, CourseLessonResultHook
from kelvin.results.services.hooks import SendHookException, send_hook

from .services.pull_triggers import pull_triggers

logger = get_task_logger(__name__)


@logged_task
@app.task(bind=True, max_retries=10)
def send_hook_task(self, course_lesson_result_hook_id):
    with transaction.atomic():
        course_lesson_result_hook = (
            CourseLessonResultHook.objects
            .select_for_update()
            .filter(id=course_lesson_result_hook_id)
            .first()
        )
        try:
            if course_lesson_result_hook:
                if not course_lesson_result_hook.send_time:
                    send_hook(
                        course_lesson_result_hook=course_lesson_result_hook,
                        commit=True,
                    )
            else:
                raise SendHookException(
                    "CourseLessonResultHook with id %s does not exist" % course_lesson_result_hook_id
                )
        except SendHookException as exc:
            raise self.retry(exc=exc, countdown=settings.CELERY_DEFAULT_COUNTDOWN)


@logged_task
@app.task(max_retries=10)
def pull_triggers_task(course_lesson_result_id):
    course_lesson_result = CourseLessonResult.objects.get(id=course_lesson_result_id)
    pull_triggers(course_lesson_result=course_lesson_result)
