from builtins import str

from kelvin.common.utils import takewhile_with_first_fail
from kelvin.lesson_assignments.models import LessonAssignment
from kelvin.lessons.models import Lesson, LessonProblemLink
from kelvin.problems.markers import Marker
from kelvin.problems.serializers import AnswerDatabaseSerializer
from kelvin.problems.utils import get_markers_dict_from_layout


def recalculate_result(result, lesson, problems_dict, assigned=None):
    """
    Общая для результата по занятию и курсозанятию функция пересчета
    Занятие, словарь задач и назначение, если оно есть, получаются внешней
    функцией в зависимости от того, какой это результат
    """
    # Рассматриваем только назначенные
    if assigned is not None:
        problems_dict = {
            str(link_id): problems_dict[str(link_id)]
            for link_id in assigned if str(link_id) in problems_dict
        }

    def make_answer_objects(user_answers):
        """Делает из словарей объекты ответов"""
        serializer = AnswerDatabaseSerializer(
            data=user_answers,
            many=True,
        )
        serializer.is_valid(raise_exception=True)
        return serializer.save()

    def marker_is_autocheckable(problem_link_id, marker_id):
        """
        Проверяет, что маркер в задаче можно проверить автоматически,
        т.е. что он в принципе проверяемый автоматически и поддерживается не
        только андроид-устройствами
        """
        problem = problems_dict[problem_link_id]
        markers = get_markers_dict_from_layout(problem.markup['layout'])
        marker = Marker(
            markers[marker_id],
            problem.markup['answers'][marker_id],
            problem.markup['checks'].get(marker_id)
        )
        return marker.checkable and not marker.is_only_android

    # Формируем объекты ответов для проверки. Игнорируем те ответы, которых
    # нет среди рассматриваемых задач из-за изменения назначения или занятия
    new_answers = {}
    ignored_answers = {}
    for problem_link_id, user_answers in result.answers.items():
        if problem_link_id in problems_dict:
            new_answers[problem_link_id] = make_answer_objects(user_answers)
        else:
            ignored_answers[problem_link_id] = user_answers

    for problem_link_id, user_answers in new_answers.items():
        # удаляем число ошибок в автопроверяемых маркерах
        for i, user_answer in enumerate(user_answers):
            # сохраняем проверенные баллы и комментарий
            user_answer.checked_points = (result.answers[problem_link_id][i]
                                          .get('checked_points'))
            user_answer.comment = (result.answers[problem_link_id][i]
                                   .get('comment', ''))

            user_answer.points = None
            user_answer.mistakes = None
            user_answer.max_mistakes = None
            for marker_id, marker_answer in (
                    user_answer.markers or {}).items():
                if marker_is_autocheckable(problem_link_id, marker_id):
                    marker_answer.pop('mistakes', None)
                    marker_answer.pop('max_mistakes', None)
                    marker_answer.pop('answer_status', None)

    # проверяем ответы
    checked_answers = Lesson.check_answers(
        problems_dict,
        new_answers,
    )

    # сериализация проверенных ответов
    # после первого правильного не сериализуем
    serializer = AnswerDatabaseSerializer(many=True)
    result.answers = {
        problem_id: serializer.to_representation(
            takewhile_with_first_fail(
                lambda problem_answer: problem_answer.mistakes,
                problem_answers,
            )
        )
        for problem_id, problem_answers in checked_answers.items()
    }
    result.max_points = lesson.get_max_points(assigned)
    result.points = lesson.get_points(result.answers)

    # Дописываем проигнорированные ответы после пересчета
    result.answers.update(ignored_answers)

    result.save()


def recalculate_lesson_result(result):
    """Пересчет результата по занятию"""
    problems_dict = {
        str(link.id): link.problem
        for link in LessonProblemLink.objects.filter(
            lesson_id=result.summary.lesson_id,
        ).select_related('problem')
    }
    return recalculate_result(result, result.summary.lesson, problems_dict)


def recalculate_clesson_result(result):
    """Пересчет результата по курсозанятию с учетом назначения"""
    problems_dict = {
        str(link.id): link.problem
        for link in LessonProblemLink.objects.filter(
            lesson_id=result.summary.clesson.lesson_id,
            problem_id__isnull=False,
        ).select_related('problem')
    }
    try:
        assigned = LessonAssignment.objects.get(
            student=result.summary.student,
            clesson=result.summary.clesson
        ).problems
    except LessonAssignment.DoesNotExist:
        assigned = None

    return recalculate_result(
        result, result.summary.clesson.lesson, problems_dict, assigned)
