import json
from builtins import str

from rest_condition import And, Not, Or

from django.conf import settings
from django.db import transaction
from django.db.utils import IntegrityError
from django.utils import timezone

from rest_framework import mixins, serializers, status, viewsets
from rest_framework.decorators import api_view, detail_route, list_route
from rest_framework.exceptions import ValidationError
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response
from rest_framework.status import HTTP_200_OK, HTTP_400_BAD_REQUEST, HTTP_404_NOT_FOUND
from rest_framework.views import APIView

from kelvin.accounts.permissions import (
    IsContentManager, IsParent, IsStaff, IsTeacher, ObjectForAuthenticated, ObjectForContentManager,
)
from kelvin.common.permissions import IsListRequest
from kelvin.common.utils import dt_to_microseconds, make_timestamp
from kelvin.courses.models import CourseLessonLink
from kelvin.courses.permissions import course_tvmservice_permission_factory
from kelvin.lesson_assignments.models import LessonAssignment
from kelvin.lessons.models import LessonProblemLink
from kelvin.problems.answers import Answer
from kelvin.problems.serializers import AnswerDatabaseSerializer, CustomAnswerSerializer
from kelvin.results.filters import CourseLessonResultFilter, LessonResultFilter
from kelvin.results.models import (
    AbstractLessonResult, CourseLessonResult, CourseLessonResultMeta, CourseLessonSummary, LessonResult, LessonSummary,
    backup_clesson_results,
)
from kelvin.results.permissions import (
    ObjectResultAuthor, ObjectResultContainsProblem, ObjectResultContainsProblemAttempt, ObjectResultForAnonymous,
)
from kelvin.results.serializers import (
    CourseLessonResultSerializer, CourseLessonSummarySerializer, LessonResultSerializer, LessonSummarySerializer,
    ResetClessonResultsInputSerializer, ResetClessonResultsInputSerializer3,
)


class LessonResultViewSet(mixins.CreateModelMixin,
                          mixins.UpdateModelMixin,
                          viewsets.ReadOnlyModelViewSet):
    """
    Просмотр и создание результатов прохождения занятий
    """
    queryset = (
        LessonResult.objects.all()
        .select_related('summary', 'summary__student', 'summary__lesson')
        .order_by('-date_created')
    )
    serializer_class = LessonResultSerializer
    permission_classes = [IsAuthenticated, ObjectResultAuthor]
    filter_class = LessonResultFilter


class CourseLessonResultViewSet(LessonResultViewSet):
    """
    Просмотр и создание результатов прохождения занятий в курсе
    """
    queryset = (
        CourseLessonResult.objects.all()
        .select_related('summary', 'summary__student', 'summary__clesson',
                        'summary__clesson__course')
        .order_by('-date_created')
    )
    serializer_class = CourseLessonResultSerializer
    permission_classes = [
        Or(
            Not(IsListRequest),
            IsParent,
            IsTeacher,
            IsContentManager,
            ObjectForContentManager,
        ),
        Or(
            And(
                ObjectForAuthenticated,
                ObjectResultAuthor,
            ),
            ObjectResultForAnonymous,
        )
    ]
    filter_class = CourseLessonResultFilter

    @list_route(permission_classes=[
        IsAuthenticated,
        IsStaff,  # FIXME temporary permission
    ])
    def versions(self, request):
        """
        Возвращает даты (версии) последнего изменения любого результата
        по занятиям курса для ученика.
        Если у урока для одного ученика два результата, вернет дату последнего
        измененного результата.
        """
        try:
            course_id = int(request.query_params.get('course'))
        except (ValueError, TypeError):
            raise ValidationError('wrong `course` parameter')

        # TODO переделать с использованием Summary
        result_dates = CourseLessonResult.objects.filter(
            summary__student=request.user,
            summary__clesson__course=course_id,
        ).order_by('date_updated').values_list(
            'summary__clesson_id', 'date_updated')

        return Response({
            clesson_id: make_timestamp(date_updated)
            for clesson_id, date_updated in result_dates
        })

    @list_route(permission_classes=[IsStaff])
    def origin_course_stat(self, request):
        """
        Статистика по оригинальному курсу по вопросам в занятии
        """
        # TODO тесты написать, когда ручкой будут пользоваться не админы
        # или хотя бы будет окончательный смысл статистик
        try:
            clesson_id = int(request.query_params.get('clesson'))
        except (TypeError, ValueError):
            raise ValidationError('wrong `clesson` parameter')

        original_clesson = (CourseLessonLink.objects
                            .select_related('lesson')
                            .prefetch_related('lesson__lessonproblemlink_set')
                            .get(id=clesson_id))
        problem_stats = {
            problem_link.problem_id: {
                'problem': problem_link.problem_id,
                'right': 0,
                'wrong': 0,
                'no_answer': 0,
                'first_attempt': 0,
                'solving': 0,
            }
            for problem_link in (original_clesson.lesson.lessonproblemlink_set
                                 .all())
            if problem_link.problem_id
        }

        # запрашиваем необходимые данные, готовим объекты для подсчета
        clessons = {}
        students = []
        problem_links_by_clesson = {}
        problem_id_by_problem_link_id = {}
        for clesson in (CourseLessonLink.objects.filter(
                        copy_of=clesson_id)
                        .select_related('course', 'lesson')
                        .prefetch_related('course__students',
                                          'lesson__lessonproblemlink_set')):
            clessons[clesson.id] = clesson
            students.extend(clesson.course.students.all())
            problem_links = (
                CourseLessonResult.get_summarizable_lessonproblemlinks(
                    clesson.lesson_id)
            )
            problem_links_by_clesson[clesson.id] = problem_links
            for problem_link in problem_links:
                problem_id_by_problem_link_id[problem_link.id] = (
                    problem_link.problem_id)
        results = {}
        for result in (CourseLessonResult.objects
                       .filter(summary__clesson__copy_of=clesson_id,
                               summary__student__in=students,
                               work_out=False)
                       .select_related('summary')):
            results[(result.summary.clesson_id, result.summary.student_id)] = (
                result)
        assignments = {}
        for assignment in (LessonAssignment.objects
                           .filter(clesson__copy_of=clesson_id,
                                   student__in=students)):
            assignments[(assignment.clesson_id, assignment.student_id)] = (
                assignment)

        # подсчет статистики
        for key, result in results.items():
            clesson_id = key[0]
            problem_links = problem_links_by_clesson[clesson_id]

            # получаем сводку результата
            if key in assignments:
                assigned_problem_link_ids = assignments[key].problems
                assigned_links = [link for link in problem_links if link.id in assigned_problem_link_ids]
                summary = AbstractLessonResult.get_summary(
                    assigned_links, result,
                    lesson_scenario=clessons[clesson_id],
                )
            else:
                summary = AbstractLessonResult.get_summary(
                    problem_links, result,
                    lesson_scenario=clessons[clesson_id],
                )

            # по сводке результата считаем статистику
            for problem_link_id, problem_summary in (
                    iter(summary['problems'].items())):
                problem_id = problem_id_by_problem_link_id[problem_link_id]
                if problem_id not in problem_stats:
                    continue
                if (not problem_summary.get('answered') and
                        not problem_summary.get('attempt_number')):
                    problem_stats[problem_id]['no_answer'] += 1
                elif problem_summary['status'] == Answer.SUMMARY_CORRECT:
                    problem_stats[problem_id]['right'] += 1
                    if problem_summary['attempt_number'] == 0:
                        problem_stats[problem_id]['first_attempt'] += 1
                elif problem_summary['status'] == Answer.SUMMARY_INCORRECT:
                    problem_stats[problem_id]['wrong'] += 1
                elif problem_summary['status'] is None:
                    problem_stats[problem_id]['solving'] += 1

        return Response(list(problem_stats.values()))

    @detail_route(methods=['post'], permission_classes=[
        IsAuthenticated, ObjectResultForAnonymous])
    def take(self, request, pk):
        """
        Проставляет текущего пользователя в попытке
        """
        result = self.get_object()
        result.summary.student = self.request.user
        try:
            result.summary.save()
        except IntegrityError:
            # у пользователя уже есть попытки по этому занятию
            return Response(status=HTTP_400_BAD_REQUEST)
        return Response(self.get_serializer().to_representation(result))

    @detail_route(
        methods=['post'],
        permission_classes=[IsAuthenticated, ObjectResultAuthor],
    )
    def add_student_viewed_problems(self, request, pk):
        """
        Принимает пары ключ-значение: problem_id(строка) - viewed(bool)
        и добавляет этот словарь в метаданные курсозанятия
        """
        student_viewed_problems = request.data.get('student_viewed_problems')
        clresult = self.get_object()
        if clresult.meta is None:
            meta_obj = CourseLessonResultMeta(
                student_viewed_problems=dict(),
            )
        else:
            meta_obj = clresult.meta

        for problem_link_id in student_viewed_problems:
            meta_obj.student_viewed_problems[str(problem_link_id)] = True

        if clresult.meta is None:
            with transaction.atomic():
                meta_obj.save()
                clresult.meta = meta_obj
                clresult.save()
        else:
            meta_obj.save()

        only_viewed_problems = [
            k for (k, v) in list(clresult.meta.student_viewed_problems.items()) if v
        ]
        return Response(
            status=HTTP_200_OK,
            data=dict({"student_viewed_problems": only_viewed_problems}),
        )

    def update(self, request, *args, **kwargs):
        """
        Могут приходить запросы по курсозанятиям, которых уже не существует,
        в этом случае отдаёт 404.
        """
        try:
            return super(CourseLessonResultViewSet, self).update(
                request, *args, **kwargs)
        except CourseLessonLink.DoesNotExist:
            raise Response(status=HTTP_404_NOT_FOUND)

    @detail_route(methods=['post'], permission_classes=[
        IsAuthenticated,
    ])
    def viewed(self, request, pk):
        """
        Выставляет флаг viewed у попытки завершенной контрольной
        """
        updated = CourseLessonResult.objects.filter(
            id=pk,
            summary__student=self.request.user,
            summary__clesson__finish_date__lte=timezone.now(),
            summary__clesson__mode=CourseLessonLink.CONTROL_WORK_MODE,
        ).update(
            viewed=True,
        )
        return Response(status=HTTP_200_OK
                        if updated else HTTP_400_BAD_REQUEST)

    @detail_route(methods=['post'], permission_classes=[
        IsAuthenticated,
        ObjectResultAuthor,
    ])
    def update_spent_time(self, request, pk):
        """
        Обновляет проведённое учеником в задаче время
        """
        result = self.get_object()
        date_updated = request.query_params.get('date_updated')

        if (
            date_updated is not None and (
                not date_updated.isdigit() or
                dt_to_microseconds(result.date_updated) != int(date_updated)
            )
        ):
            # переданный в GET параметре date_updated
            # должен соответствовать предыдущему значению
            return Response(status=HTTP_404_NOT_FOUND)

        time_delta = request.data.get('time_delta')
        link_id = request.data.get('link_id')

        if time_delta is None or link_id is None:
            # обязательные поля
            return Response(status=HTTP_400_BAD_REQUEST)

        clesson = result.summary.clesson
        # контрольная уже закончилась
        if (clesson.mode in CourseLessonLink.EVALUATION_LESSON_MODES and
                timezone.now() >= result.quiz_time_limit(clesson)):
            return Response(status=HTTP_400_BAD_REQUEST)

        link_exists = LessonProblemLink.objects.filter(
            id=int(link_id),
            lesson_id=clesson.lesson_id,
        ).exists()

        if not link_exists:
            # попытка обновить время по несуществующей связи
            return Response(status=HTTP_400_BAD_REQUEST)

        time_delta = int(time_delta)

        if time_delta <= 0:
            # дельта времени должна быть положительной
            return Response(status=HTTP_400_BAD_REQUEST)

        result.spent_time = (result.spent_time or 0) + time_delta

        answers = result.answers.get(str(link_id))
        answer_serializer = AnswerDatabaseSerializer()

        # ToDo: изменяем время по уже завершенной попытке. Ждем поддержки
        # функционала от фронта
        if answers:
            last_answer_data = answers[-1]
            current_spent_time = last_answer_data.get('spent_time') or 0
            last_answer_data['spent_time'] = current_spent_time + time_delta
            last_answer = Answer(**last_answer_data)
            answers[-1] = answer_serializer.to_representation(last_answer)
        else:
            last_answer = Answer(
                completed=False,
                markers=None,
                theory=None,
                spent_time=time_delta,
            )
            answers = [answer_serializer.to_representation(last_answer)]

        result.answers[str(link_id)] = answers
        result.save()

        return Response(
            data=self.get_serializer().to_representation(result),
            status=HTTP_200_OK,
        )


class CourseLessonCommentViewSet(viewsets.GenericViewSet):
    """
    Создание и изменение комментариев к попыткам
    """

    queryset = CourseLessonResult.objects.all()

    permission_classes = [
        IsAuthenticated,
        ObjectResultContainsProblem,
        ObjectResultContainsProblemAttempt,
    ]

    @detail_route(url_path='problem_link/(?P<problem_link_id>[0-9]+)/comment')
    def post(self, request, pk, problem_link_id, *args, **kwargs):
        """
        Добавляет новый комментарий в существующую попытку
        """
        course_lesson_result = self.get_object()

        # Создаем сообщение
        message = self.create_validate_message(request.body)

        # Берем последнюю попытку по задаче
        last_attempt = course_lesson_result.answers[problem_link_id][-1]

        # Добавляем в нее новое сообщение
        if ('custom_answer' not in last_attempt or
                last_attempt['custom_answer'] is None):
            last_attempt['custom_answer'] = [message]
        else:
            last_attempt['custom_answer'].append(message)

        self._create_or_update_meta(problem_link_id)
        return Response(self.validate_results(course_lesson_result, message))

    @detail_route(url_path='problem_link/(?P<problem_link_id>[0-9]+)/comment')
    def patch(self, request, pk, problem_link_id, *args, **kwargs):
        """
        Изменяет последний комментарий
        """
        course_lesson_result = self.get_object()
        message = self.create_validate_message(request.body)
        last_attempt = course_lesson_result.answers[problem_link_id][-1]

        if ('custom_answer' not in last_attempt or
                last_attempt['custom_answer'] is None):
            last_attempt['custom_answer'] = []

        # Изменять можно только свое сообщение
        if message['user']['id'] != request.user.id:
            raise ValidationError(u'Изменять можно только свое сообщение')

        # Нельзя менять тип сообщения
        if (message['type'] !=
                last_attempt['custom_answer'][-1]['type']):
            raise ValidationError(u'Нельзя менять тип сообщения')

        # Если сообщений еще нет, то патчить нельзя
        if not last_attempt['custom_answer']:
            raise ValidationError(u'Если сообщений еще нет, то патчить '
                                  u'нельзя')

        last_attempt['custom_answer'][-1] = message
        self._create_or_update_meta(problem_link_id)
        return Response(self.validate_results(course_lesson_result, message))

    def validate_results(self, course_lesson_result, message):
        """
        Функция принимает clesson_result пересчитывает для него результаты,
        сохраняет и возвращает новые данные
        """
        context = self.get_serializer_context()
        context['update_only_custom_answer'] = message
        serializer = CourseLessonResultSerializer(
            course_lesson_result,
            data=CourseLessonResultSerializer(course_lesson_result).data,
            context=context,
        )
        serializer.is_valid(raise_exception=True)
        serializer.save()
        return serializer.data

    def create_validate_message(self, message_string):
        """
        Метод создает, валидирует, проставляет нужные поля у объекта сообщения
        из строки
        """
        message = json.loads(message_string)
        message_serializer = CustomAnswerSerializer(
            data=message,
            context=self.get_serializer_context()
        )
        message_serializer.is_valid(raise_exception=True)
        return message_serializer.data

    def _create_or_update_meta(self, problem_link_id):
        course_lesson_result = self.get_object()
        if course_lesson_result.meta is None:
            meta_obj = CourseLessonResultMeta.objects.create(
                student_viewed_problems={problem_link_id: False},
            )
            course_lesson_result.meta = meta_obj
        else:
            meta_obj = course_lesson_result.meta
            meta_obj.student_viewed_problems[problem_link_id] = False
            meta_obj.save()


class LessonSummaryViewSet(viewsets.ReadOnlyModelViewSet):
    """
    Просмотр сводки результатов по занятию
    """
    queryset = LessonSummary.objects.all()
    serializer_class = LessonSummarySerializer
    permission_classes = [
        IsAuthenticated,
        IsStaff,  # FIXME temporary permission
    ]
    filter_fields = (
        'lesson',
        'student',
    )


class CourseLessonSummaryViewSet(viewsets.ReadOnlyModelViewSet):
    """
    Просмотр сводки результатов по занятию в курсе
    """
    queryset = CourseLessonSummary.objects.all()
    serializer_class = CourseLessonSummarySerializer
    permission_classes = [
        IsAuthenticated,
        IsStaff,  # FIXME temporary permission
    ]
    filter_fields = (
        'clesson',
        'student',
    )


class ResetClessonResults(APIView):
    """Удаление результатов пользователя по курсу и модулю."""

    input_serializer = ResetClessonResultsInputSerializer

    def delete(self, request):
        if not request.user.has_perm("results.backup_task_status"):
            return Response(status=status.HTTP_403_FORBIDDEN)

        input = self.input_serializer(data=request.query_params)
        input.is_valid(raise_exception=True)

        queryset = CourseLessonResult.objects.filter(
            summary__student__username=input.data["username"],
            summary__clesson__course_id=input.data["course_id"],
        )
        if "clesson_id" in input.data:
            queryset = queryset.filter(
                summary__clesson_id=input.data["clesson_id"]
            )
        result_ids = queryset.values_list("id", flat=True)
        backup_clesson_results(result_ids)

        return Response(status=status.HTTP_200_OK)


class ResetClessonResultsV3(ResetClessonResults):
    """Удаление результатов пользователя по курсу и модулю."""

    input_serializer = ResetClessonResultsInputSerializer3
    authentication_classes = []
    permission_classes = [
        course_tvmservice_permission_factory(
            course_id_attr_name='course_id',
            attr_path='request.data',
        ),
    ]

    def delete(self, request):
        input = self.input_serializer(data=request.data)
        input.is_valid(raise_exception=True)

        queryset = CourseLessonResult.objects.filter(
            summary__student__username=input.data["username"],
            summary__clesson__course_id=input.data["course_id"],
        )
        if "clesson_id" in input.data:
            queryset = queryset.filter(
                summary__clesson_id=input.data["clesson_id"]
            )
        queryset.delete()

        return Response(status=status.HTTP_204_NO_CONTENT)
