from builtins import object

from django.conf import settings
from django.db import models
from django.db.models import Subquery
from django.utils.translation import ugettext_lazy as _

from kelvin.common.avatars import get_avatar
from kelvin.common.fields import JSONField
from kelvin.common.model_mixins import TimeStampMixin
from kelvin.common.revisor import client as revisor
from kelvin.courses.models import CourseLessonLink
from kelvin.problems.models import Problem
from kelvin.results.models import CourseLessonResult
from kelvin.reviews.exceptions import RevisorSyncException


class ReviewTask(TimeStampMixin, models.Model):
    """
    ReviewTask user data in courselesson
    """
    data = JSONField(
        verbose_name=_(u'Данные'),
        default=dict,
    )
    student_data = JSONField(
        verbose_name=_(u'Пометки студента о ревью'),
        default=dict,
    )
    external_data = JSONField(
        verbose_name=_(u"Внешняя информация для интгерации"),
        default=dict,
    )
    problem = models.ForeignKey(
        Problem,
        verbose_name=_(u'Задача'),
    )
    student = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        verbose_name=_(u'Ученик'),
    )

    class Meta(object):
        verbose_name = _(u'Работа ученика на ревью')
        verbose_name_plural = _(u'Работы учеников на ревью')

    def save(self, push_to_revisor=True, **kwargs):
        if push_to_revisor:
            self.push_to_revisor()
        super(ReviewTask, self).save(**kwargs)

    def push_to_revisor(self):
        """Send ReviewTask to Revisor for review."""
        group_id = self.problem.external_data.get('revisor_group_id')
        if not group_id:
            raise RevisorSyncException('revisor_group_id not found')

        # Добавляем дополнительной инфы для отображения в интерфейсе ревьюера напрямую из revisor
        self.data['author'] = {
            'avatar': get_avatar(self.student.username, 64),
            'name': self.student.get_full_name(),
        }
        self.data['markup'] = self.problem.markup

        # Добавляем информацию о курсе и модуле, предполагаем, что этот problem только в одном lesson и в одном course
        lesson_problem = self.problem.lessonproblemlink_set.select_related('lesson').first()
        if lesson_problem:
            self.data['lesson_name'] = lesson_problem.lesson.name
            clesson = CourseLessonLink.objects.select_related('course').filter(lesson=lesson_problem.lesson).first()
            if clesson:
                self.data['course_name'] = clesson.course.name

        if self.external_data.get('revisor_task'):
            task = revisor.update_task(
                revisor_task_id=self.external_data['revisor_task']['id'],
                data=self.data,
                status='new',
            )
        else:
            task = revisor.create_task(
                group_id=group_id,
                data=self.data,
                thread_id='student={};problem={}'.format(self.student_id, self.problem_id),
            )
        self.external_data['revisor_task'] = task

    def update_course_lesson_results(self):
        clesson_results = CourseLessonResult.objects.filter(
            summary__clesson__lesson_id__in=Subquery(
                self.problem.lessonproblemlink_set.values_list('lesson_id')
            ),
            summary__student=self.student,
        )

        for clesson_result in clesson_results:
            revisor_task = self.external_data['revisor_task']
            if revisor_task['status'] == 'done' and revisor_task['data'].get('accepted'):
                clesson_result.points += 1
                clesson_result.save()

    def fetch_from_revisor(self):
        revisor_task = self.external_data.get('revisor_task')
        if not revisor_task:
            raise RevisorSyncException('revisor_task_id not found')

        # если таск уже в статусе 'done', то он не может измениться в revisor, можем не обновлять его
        if revisor_task['status'] == 'done':
            return

        task = revisor.get_task(revisor_task['id'])
        self.external_data['revisor_task'] = task
        self.data = task['data']
        if task['status'] == 'done':
            self.update_course_lesson_results()
        self.save(push_to_revisor=False)

    def get_revisor_history(self):
        revisor_task = self.external_data.get('revisor_task')
        if not revisor_task:
            return []

        thread_id = revisor_task.get('thread_id')
        if not thread_id:
            return []

        return revisor.get_tasks(thread_id=thread_id)
