from builtins import object

from rest_framework import serializers

from kelvin.accounts.models import User
from kelvin.reviews.exceptions import RevisorSyncException
from kelvin.reviews.models import ReviewTask


class ReviewTaskSerializer(serializers.ModelSerializer):
    """
    ReviewTask model serializer
    """
    student = serializers.IntegerField(
        required=False,
        write_only=True,
        default=serializers.CurrentUserDefault(),
    )
    status = serializers.SerializerMethodField()
    modified_at = serializers.SerializerMethodField()

    class Meta(object):
        model = ReviewTask
        fields = ('data', 'student_data', 'student', 'problem', 'status', 'modified_at')

    def get_status(self, review_task):
        status = 'new'
        revisor_task = review_task.external_data.get('revisor_task')

        if revisor_task and 'status' in revisor_task:
            status = revisor_task['status']

        return status

    def get_modified_at(self, review_task):
        modified_at = None
        revisor_task = review_task.external_data.get('revisor_task')

        if revisor_task and 'modified_at' in revisor_task:
            modified_at = revisor_task['modified_at']

        return modified_at

    def validate(self, attrs):
        problem = attrs['problem']
        student = self.context['request'].user

        current_task = ReviewTask.objects.filter(problem=problem, student=student).order_by('id').last()
        if not current_task:
            # Это первый таск в своём трэде, можно просто создавать его
            return attrs
        try:
            current_task.fetch_from_revisor()
            status = current_task.external_data['revisor_task']['status']
        except RevisorSyncException:
            # Если не удалось достать таск из ревизора, то считаем, что его статус failed.
            # Значит позволим пользователю обновить его.
            status = 'failed'

        if status == 'new' or status == 'failed':
            # Этот грязный хак позволяет нам обновить current_task вместо создания нового объекта
            self.instance = current_task
        elif status == 'in_progress':
            # Если ревью уже в работе, то запрещаем его обновлять или создавать новую версию
            raise serializers.ValidationError("Review already in_progress")
        elif status == 'done' and current_task.external_data['revisor_task']['data'].get('accepted'):
            raise serializers.ValidationError("Review already accepted")

        return attrs

    def validate_data(self, value):
        """
        Check if `data` field is dict
        """
        if not isinstance(value, dict):
            raise serializers.ValidationError("Data must be dict")
        return value

    def validate_student_data(self, value):
        """
        Check if `student_data` field is dict
        """
        if not isinstance(value, dict):
            raise serializers.ValidationError("Data must be dict")
        return value


class ReviewTaskWithHistorySerializer(ReviewTaskSerializer):
    """
    ReviewTaskWithHistorySerializer serializer
    """
    class Meta(ReviewTaskSerializer.Meta):
        fields = ReviewTaskSerializer.Meta.fields + ('history', )

    history = serializers.SerializerMethodField(read_only=True)

    def get_history(self, review_task):
        history = ReviewTask.objects.filter(problem=review_task.problem, student=review_task.student).order_by('-id')
        return ReviewTaskSerializer(
            history[1:],  # Отрезаем из истории сам текущий таск
            many=True,
        ).data


class ReviewTaskUpdateSerializer(serializers.Serializer):
    """
    ReviewTaskUpdateSerializer model serializer
    """
    student_data = serializers.DictField()
