import logging

from django_replicated.decorators import use_state

from django.http import Http404
from django.shortcuts import get_object_or_404
from django.utils.decorators import method_decorator

from rest_framework import mixins, permissions, viewsets
from rest_framework.response import Response

from kelvin.problems.models import Problem
from kelvin.reviews.models import ReviewTask
from kelvin.reviews.serializers import ReviewTaskUpdateSerializer, ReviewTaskWithHistorySerializer

# fix default django_replicated decorators
use_master = use_state(forced_state='master')


logger = logging.getLogger(__name__)


class ObjectReviewTaskDataAuthor(permissions.BasePermission):
    """
    Check if user is student that created review task data
    """
    def has_object_permission(self, request, view, obj):
        return request.user.id == obj.student_id


class ReviewTaskViewSet(
    mixins.RetrieveModelMixin,
    mixins.CreateModelMixin,
    mixins.UpdateModelMixin,
    viewsets.GenericViewSet
):
    """
    Retrieve and create review task user data in problem
    """
    lookup_value_regex = '\d+'
    queryset = ReviewTask.objects.all()
    serializer_class = ReviewTaskWithHistorySerializer
    permission_classes = [permissions.IsAuthenticated, ObjectReviewTaskDataAuthor]

    def get_object(self):
        """
        Get object by problem and student_id.
        """
        queryset = self.get_queryset()
        problem = get_object_or_404(
            Problem.objects.all(),
            pk=self.kwargs['pk'],
        )

        obj = queryset.filter(
            problem=problem,
            student=self.request.user,
        ).last()
        if not obj:
            raise Http404('No %s matches the given query.' % queryset.model._meta.object_name)
        obj.fetch_from_revisor()

        self.check_object_permissions(self.request, obj)
        return obj

    @method_decorator(use_master)
    def dispatch(self, request, *args, **kwargs):
        return super(ReviewTaskViewSet, self).dispatch(request, *args, **kwargs)

    def update(self, request, *args, **kwargs):
        instance = self.get_object()
        serializer = ReviewTaskUpdateSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        instance.student_data = serializer.data['student_data']
        instance.save(push_to_revisor=False)

        return Response(serializer.data)
