import logging
from builtins import object

from rest_framework import serializers

from kelvin.common.serializer_mixins import DateUpdatedFieldMixin, SerializerManyToManyMixin
from kelvin.courses.serializers.course_lessons_list import CourseLessonsListSerializer
from kelvin.lessons.models import Lesson, LessonProblemLink
from kelvin.problems.models import Problem
from kelvin.result_stats.models import StudentCourseStat
from kelvin.results.models import CourseLessonResult
from kelvin.subjects.models import Subject

logger = logging.getLogger(__name__)


class CourseDetailClessonProblemsSerializer(serializers.ModelSerializer):
    """
    Сериализатор заданий урока
    """
    class Meta(object):
        model = Problem
        fields = (
            'id',
            'name',
        )


class SiriusCourseLessonResultSerializer(serializers.ModelSerializer):
    """
    Сериализатор результатов прохождения курсозанятия
    """
    problems_count = serializers.SerializerMethodField()
    clesson = CourseLessonsListSerializer(source='summary.clesson')

    class Meta(object):
        model = CourseLessonResult
        fields = (
            'points',
            'max_points',
            'problems_count',
            'clesson',
        )

    def get_problems_count(self, course_lesson_result):
        """
        Возвращает количество заданий в курсозанятии
        """
        return course_lesson_result.summary.clesson.lesson.problems.count()


class SiriusCourseLessonResultsGroupSerializer(serializers.ModelSerializer):
    """
    Сериализатор результатов по курсу
    (прокидываем то, что нужно на страницу курса с модулями)
    """
    clesson_id = serializers.IntegerField(source='summary.clesson.id')
    finish_date = serializers.DateTimeField(
        source='summary.clesson.finish_date',
    )
    evaluation_date = serializers.DateTimeField(
        source='summary.clesson.evaluation_date',
    )

    class Meta(object):
        model = CourseLessonResult
        fields = (
            'id',
            'points',
            'clesson_id',
            'finish_date',
            'evaluation_date',
            'completed',
            'date_created',
            'answers',
        )


class GetClessonFeaturesApiSerializer(serializers.Serializer):
    """
    Сериализатор курсозанятий с общими данными по списку курсов
    (points, max_points, started_at) для API
    """
    _cached_clesson_reqults_qs = None

    def __init__(self, courses_list, user_id):
        self._courses_list = courses_list
        self._user_id = user_id

    def get_clessons_features(self):
        max_points = self.get_max_points(self._courses_list)
        started_at = self.get_started_at(self._courses_list)

        return [max_points, started_at]

    def _cache_clesson_results_qs(self, courses_list):
        """
        Кеширует CourseLessonResult queryset
        """
        if self._cached_clesson_reqults_qs is None:
            self._cached_clesson_reqults_qs = (
                CourseLessonResult.objects.filter(
                    summary__clesson__course__in=courses_list,
                    summary__student=self._user_id,
                    work_out=False
                )
                .select_related(
                    'summary',
                    'summary__clesson'
                )
            )

    def get_max_points(self, courses_list):
        """
        Возвращает максимальное количество баллов
        """
        qs = LessonProblemLink.objects.filter(
            type=LessonProblemLink.TYPE_COMMON,
            lesson__course__in=courses_list
        ).select_related(
            'problem',
            'lesson'
        )

        clesson_max_points = {}
        processed_clessons_id = []

        for lpl in qs:
            if lpl.id in processed_clessons_id:
                continue
            else:
                processed_clessons_id.append(lpl.id)

            key = lpl.lesson_id
            if (
                lpl.options is None or
                getattr(lpl, 'options', {}).get('max_points') is None
            ):
                if lpl.problem:
                    clesson_max_points[key] = (
                        clesson_max_points.get(key, 0) + lpl.problem.max_points
                    )
            else:
                clesson_max_points[key] = (
                    clesson_max_points.get(key, 0) +
                    lpl.options.get('max_points')
                )

        return clesson_max_points

    def get_started_at(self, courses_list):
        """
        Возвращает время и дату начала выполнения курсозанятия
        """
        self._cache_clesson_results_qs(courses_list)

        clesson_results_list = list(self._cached_clesson_reqults_qs.all())

        date_data = {}
        date_format = serializers.DateTimeField().to_representation
        for clesson_result in clesson_results_list:
            date_data[clesson_result.summary.clesson.id] = date_format(
                clesson_result.date_created) if clesson_result else None

        return date_data


class LessonInCLessonSerializer(DateUpdatedFieldMixin,
                                SerializerManyToManyMixin,
                                serializers.ModelSerializer):
    """
    Сериализатор для курсозанятия без сценария
    """
    subject = serializers.CharField(source='subject.slug', required=False,
                                    read_only=True)

    m2m_update_fields = {
        'problems': 'lesson',
    }

    class Meta(object):
        model = Lesson
        fields = (
            'id',
            'date_updated',
            'name',
            'theme',
            'subject',
        )
        read_only_fields = (
            'date_updated',
            'subject',
        )

    def validate(self, attrs):
        """Проставляем порядок в связях"""
        if 'lessonproblemlink_set' in attrs:
            for i, problem_link in enumerate(
                    attrs['lessonproblemlink_set'], 1):
                problem_link['order'] = i
        return super(LessonInCLessonSerializer, self).validate(attrs)


class CourseSubjectSerializer(serializers.ModelSerializer):
    """
    Сериализатор учебных предметов
    """

    class Meta(object):
        model = Subject
        fields = ('id', 'slug', 'name')
        read_only_fields = ('id', 'slug', 'name')


class CLessonProblemSerializer(serializers.ModelSerializer):
    """
    Сериализатор задания урока
    """
    name = serializers.SerializerMethodField()
    max_points = serializers.IntegerField(source='problem.max_points',
                                          read_only=True)

    class Meta(object):
        model = LessonProblemLink
        fields = (
            'id',
            'name',
            'order',
            'max_points',
            'type',
        )
        read_only_fields = (
            'id',
            'name',
            'order',
            'max_points',
            'type',
        )

    def get_name(self, lesson_problem_link):
        """
        Возвращает имя задачи или теории
        """
        if lesson_problem_link.theory:
            return lesson_problem_link.theory.name

        return lesson_problem_link.problem.name


class SiriusStudentCourseStatSerializer(serializers.ModelSerializer):
    """
    Сериализатор для выгрузки данных о результатах прохождления
    курса всеми пользователями во внешнюю систему аналитики
    """
    username = serializers.CharField(source='student.username')

    class Meta(object):
        model = StudentCourseStat
        fields = (
            'username',
            'points',
        )


class CourseLessonResultExportSerializer(serializers.ModelSerializer):
    """
    Сериализатор выгрузки результатов занятия
    """
    username = serializers.CharField(source='summary__student__username')
    points = serializers.IntegerField()

    class Meta(object):
        model = CourseLessonResult
        fields = (
            'username',
            'points',
        )


class SiriusStudentCourseStatByUsernameSerializer(serializers.Serializer):
    """
    Сериализатор для выгрузки данных о результатах прохождения
    списка курсов пользователем во внешнюю систему аналитики
    """
    course_id = serializers.IntegerField()
    points = serializers.IntegerField()

    class Meta(object):
        fields = (
            'course_id',
            'points',
        )
