from builtins import map

from rest_framework import status, viewsets
from rest_framework.decorators import detail_route
from rest_framework.permissions import IsAdminUser, IsAuthenticated
from rest_framework.response import Response

from kelvin.common.utils import LoggableMixin, log_method
from kelvin.result_stats.models import StudentCourseStat
from kelvin.sapi.pagination import SiriusExportPaginator
from kelvin.sapi.serializers import SiriusStudentCourseStatByUsernameSerializer, SiriusStudentCourseStatSerializer


class SiriusStudentCourseStat(LoggableMixin, viewsets.ReadOnlyModelViewSet):
    """
    API результатов пройденных курсов
    """
    queryset = StudentCourseStat.objects.all().select_related('student')
    serializer_class = SiriusStudentCourseStatSerializer
    pagination_class = SiriusExportPaginator

    permission_classes = [
        IsAdminUser,
    ]

    @log_method
    @detail_route(methods=['get'])
    def all_students(self, request, pk=None):
        """
        Оверрайд метода получания единичного объекта
        """
        qs = self.queryset.filter(course=pk)

        page_objects = self.paginate_queryset(qs)
        serializer = SiriusStudentCourseStatSerializer(page_objects, many=True)

        return self.paginator.get_paginated_response(serializer.data)

    @log_method
    @detail_route(methods=['get'], permission_classes=[
        IsAuthenticated,
    ])
    def user_results(self, request, pk=None):
        """
        Возвращает список результатов пользователя по списку id курсов
        """
        raw_course_ids = self.request.query_params.get('course_ids', '')
        course_ids = raw_course_ids.split(',')

        try:
            course_ids = list(map(int, course_ids))
        except ValueError:
            return Response(data='course_ids must be set and be integers',
                            status=status.HTTP_400_BAD_REQUEST)

        qs = self.queryset.filter(
            course__in=course_ids,
            student__username=pk
        ).values(
            'course_id',
            'points',
        )

        page_objects = self.paginate_queryset(qs)
        serializer = SiriusStudentCourseStatByUsernameSerializer(
            page_objects, many=True
        )

        return self.paginator.get_paginated_response(serializer.data)
