import json
from builtins import range, str

from django import forms
from django.utils.translation import ugettext_lazy as _

from kelvin.courses.models import Course, CourseLessonLink
from kelvin.scorm.models import Scorm

KEYS_DELIMITER = '->'


def extract_full_keys(data, delimiter):
    """
    Get all keys inside data recursively. Data may be list or dict.
    Keys will be presented as strings so int(float,...) will be lost.
    """
    extracted_keys = []

    def extract_keys(data, extracted_keys=list(), cur_key=''):
        if isinstance(data, dict):
            for key in data:
                extract_keys(
                    data[key],
                    extracted_keys,
                    cur_key + delimiter + str(key) if cur_key else str(key),
                )
        elif isinstance(data, list):
            for i in range(len(data)):
                extract_keys(
                    data[i],
                    extracted_keys,
                    cur_key + delimiter + str(i) if cur_key else str(i),
                )
        else:
            extracted_keys.append(cur_key)

    extract_keys(data, extracted_keys)
    return extracted_keys


def get_value_by_full_key(data, full_key, delimiter):
    """
    Get value from recursively generated full_key with delimiter.
    Available only string keys (JSON).
    """
    keys = full_key.split(delimiter)
    res_value = data

    try:
        for key in keys:
            if isinstance(res_value, list):
                res_value = res_value[int(key)]
            else:
                res_value = res_value[key]
    except (IndexError, KeyError):
        return '-'
    return res_value


def scorm_csv_generator(students_scorm_data_list, scorm_keys):
    """
    Get data from list, than get all values by scorm full_keys and
    return it by rows. One row => one student scorm data.
    First row contains all keys for column title.
    """
    yield scorm_keys
    for scorm_data in students_scorm_data_list:
        row = []
        for full_key in scorm_keys:
            value = get_value_by_full_key(scorm_data, full_key, KEYS_DELIMITER)
            if isinstance(value, str):
                value = value.encode('utf-8')
            else:
                value = str(value).encode('utf-8')
            row.append(value)
        yield row


def scorm_json_generator(students_scorm_data_list):
    """
    Get all users data from list of scorm users results and
    yield it by one result. Complete return is json array of results.
    """
    output = '['
    for data in students_scorm_data_list:
        yield output
        output = json.dumps(data) + ','
    yield output[:-1] + ']'


class ScormResultDownloadForm(forms.Form):
    """
    Form to download all scorm results in course
    """
    scorm_clessons = Scorm.objects.values_list('clesson', flat=True)
    scorm_courses = CourseLessonLink.objects.filter(
        id__in=scorm_clessons).values_list('course_id', flat=True)

    course = forms.ModelChoiceField(
        queryset=Course.objects.filter(id__in=scorm_courses),
        label=_(u'Курс SCORM'),
        required=True,
    )
    export_type = forms.ChoiceField(
        choices=(
            ('csv', 'CSV'),
            ('json', 'JSON'),
        ),
        label=_(u'Тип выгрузки'),
        required=True,
    )

    def export(self):
        """
        Get users data from scorm and return generator, that yield rows with
        scorm data of each student.
        """
        course = self.cleaned_data.get('course')
        export_type = self.cleaned_data.get('export_type')
        clesson = CourseLessonLink.objects.filter(
            id__in=self.scorm_clessons,
            course=course,
        )

        students_scorm_data_list = (
            Scorm.objects.filter(clesson=clesson).values_list(
                'data',
                flat=True,
            )
        )
        if students_scorm_data_list is None or not export_type:
            return

        if export_type == 'csv':
            scorm_full_keys_set = set()
            for student_scorm_data in students_scorm_data_list:
                scorm_data_keys = (
                    extract_full_keys(
                        student_scorm_data,
                        KEYS_DELIMITER,
                    )
                )
                for scorm_data_key in scorm_data_keys:
                    scorm_full_keys_set.add(scorm_data_key)

            scorm_keys = sorted(list(scorm_full_keys_set))

            return (
                scorm_csv_generator(students_scorm_data_list, scorm_keys),
                export_type,
            )
        elif export_type == 'json':
            return (
                scorm_json_generator(students_scorm_data_list),
                export_type,
            )
