import csv
import datetime
from builtins import object

from simple_history.admin import SimpleHistoryAdmin

from django.conf import settings
from django.conf.urls import url
from django.contrib import admin
from django.http import StreamingHttpResponse
from django.template.response import TemplateResponse

from kelvin.courses.models import Course
from kelvin.scorm.admin.forms import ScormResultDownloadForm
from kelvin.scorm.models import Scorm, ScormISpringSuiteData, ScormISpringSuiteQuestionsData, ScormResourceUserData


class Echo(object):
    """
    An object that implements just the write method of the file-like
    interface.
    """
    def write(self, value):
        """Write the value by returning it, instead of storing in a buffer."""
        return value


@admin.register(Scorm)
class ScormAdmin(SimpleHistoryAdmin):
    """
    Scorm admin
    """
    change_list_template = 'scorm/admin/change_list.html'
    results_download_template = 'scorm/admin/results_download_form.html'

    date_hierarchy = 'date_updated'
    list_display = (
        'id',
        'clesson',
        'student',
        'date_updated',
        'date_created',
    )
    raw_id_fields = (
        'clesson',
        'student',
    )
    history_list_display = (
        'score_percent',
        'passed',

    )

    def get_urls(self):
        """
        Add url for download scorm results
        """
        return [
            url(r'results_download/$',
                self.admin_site.admin_view(self.results_download),
                name='scorm_scorm_results_download'),
        ] + super(ScormAdmin, self).get_urls()

    def results_download(self, request):
        """
        Return streaming response with csv or json, that contains students
        scorm data in one course (clesson).
        """
        form = ScormResultDownloadForm(data=request.POST)
        scorm_results_generator, export_type = (
            form.export() if form.is_valid()
            else ([], None)
        )
        context = {
            'opts': self.model._meta,
            'form': form,
        }

        if request.method == 'POST' and scorm_results_generator:
            response = None
            if export_type == 'csv':
                pseudo_buffer = Echo()
                writer = csv.writer(pseudo_buffer)
                response = StreamingHttpResponse(
                    (writer.writerow(row) for row in scorm_results_generator),
                    content_type='text/csv',
                )
            elif export_type == 'json':
                response = StreamingHttpResponse(
                    scorm_results_generator,
                    content_type='application/json',
                )

            if response:
                response['Content-Disposition'] = (
                    'attachment; filename="{course}_{date}.{format}"'.format(
                        course=Course.objects.get(
                            id=request.POST.get('course'),
                        ),
                        date=datetime.datetime.now().strftime(
                            settings.DATETIME_FORMAT,
                        ),
                        format=export_type,
                    )
                )
                return response

        return TemplateResponse(
            request,
            self.results_download_template,
            context,
        )


@admin.register(ScormResourceUserData)
class ScormResourceUserDataAdmin(SimpleHistoryAdmin):
    date_hierarchy = 'date_updated'
    list_display = (
        'id',
        'clesson',
        'student',
        'resource_ref',
        'date_updated',
        'date_created',
    )
    raw_id_fields = (
        'clesson',
        'student',
    )


@admin.register(ScormISpringSuiteData)
class ScormISpringSuiteDataAdmin(admin.ModelAdmin):
    list_display = ('scorm', 'qt')
    raw_id_fields = (
        'scorm',
    )


@admin.register(ScormISpringSuiteQuestionsData)
class ScormISpringSuiteQuestionsDataAdmin(admin.ModelAdmin):
    list_display = ('scorm', 'qt')
    raw_id_fields = (
        'scorm',
    )
