ISPRING_REPORT_FIELDS_MAP = {
    'quizSettings.@quizType': 'quiz_type',
    'quizSettings.@maxScore': 'max_score',
    'quizSettings.@maxNormalizedScore': 'max_normalized_score',
    'quizSettings.@timeLimit': 'time_limit',
    'quizSettings.passingPercent': 'passing_percent',
    'summary.@score': 'score',
    'summary.@percent': 'percent',
    'summary.@time': 'time',
    'summary.@finishTimestamp': 'finish_timestamp',
    'summary.@passed': 'passed',
}


def lookup_field(item, field, default=None):
    keys = field.split('.')
    value = item
    for k in keys:
        value = value.get(k, None) if isinstance(value, dict) else None
        if value is None:
            return default

    return value


def parse_multipleResponseQuestion(question):
    return {
        'answers': [answer['text'] for answer in question['answers']['answer']],
        'user_answer': [answer['text'] for answer in question['answers']['answer'] if answer['@correct'] == 'true'],
        'correct_answer': [answer['text'] for answer in question['answers']['answer'] if answer['@selected'] == 'true'],
    }


def parse_trueFalseQuestion(question):
    return {
        'answers': [answer['text'] for answer in question['answers']['answer']],
        'user_answer': question['answers']['answer'][int(question['answers']['@userAnswerIndex'])]['text'],
        'correct_answer': question['answers']['answer'][int(question['answers']['@correctAnswerIndex'])]['text']
    }


def parse_typeInQuestion(question):
    return {
        'answers': None,
        'user_answer': question['@userAnswer'],
        'correct_answer': question['acceptableAnswers']['answer']
    }


def parse_numericQuestion(question):
    return {
        'answers': None,
        'user_answer': question['@userAnswer'],
        'correct_answer': question['answers']
    }


def parse_sequenceQuestion(question):
    return {
        'answers': None,
        'user_answer': [answer['text'] for answer in question['answers']['answer']],
        'correct_answer': None,
    }


def parse_matchingQuestion(question):
    premises = [premise['text'] for premise in question['premises']['premise']]
    responses = [response['text'] for response in question['responses']['response']]
    return {
        'answers': None,
        'user_answer': [
            (premises[int(match_item['@premiseIndex'])], responses[int(match_item['@responseIndex'])])
            for match_item
            in question['userAnswer']['match']
        ],
        'correct_answer': [
            (premises[int(match_item['@premiseIndex'])], responses[int(match_item['@responseIndex'])])
            for match_item
            in question['matches']['match']
        ],
    }


def parse_fillInTheBlankQuestion(question):
    return {
        'answers': None,
        'user_answer': [blank_item['@userAnswer'] for blank_item in question['details']['blank']],
        'correct_answer': [blank_item['answer'] for blank_item in question['details']['blank']],
    }


def parse_multipleChoiceTextQuestion(question):
    variants = question['details']['blank']['answer']
    return {
        'answers': variants,
        'user_answer': variants[int(question['details']['blank']['@userAnswerIndex'])],
        'correct_answer': variants[int(question['details']['blank']['@correctAnswerIndex'])],
    }

def parse_wordBankQuestion(question):
    return {
        'answers': None,
        'user_answer': [blank_item['@userAnswer'] for blank_item in question['details']['blank']],
        'correct_answer': [blank_item['#text'] for blank_item in question['details']['blank']],
    }


def parse_likertScaleQuestion(question):
    statements = [statements['text'] for statements in question['statements']['statement']]
    scale_labels = question['scaleLabels']['label']
    return {
        'answers': None,
        'user_answer': [
            (statements[int(match_item['@statementIndex'])], scale_labels[int(match_item['@labelIndex'])])
            for match_item
            in question['userAnswer']['match']
        ],
        'correct_answer': None
    }


def parse_multipleChoiceQuestion(question):
    answers = [answer['text'] for answer in question['answers']['answer']]
    return {
        'answers': answers,
        'user_answer': answers[int(question['answers']['@userAnswerIndex'])],
        'correct_answer': answers[int(question['answers']['@correctAnswerIndex'])],
    }


def parse_dndQuestion(question):
    destination_map = {
        i: destination['#text']
        for i, destination
        in enumerate(question['destinations']['destination'])
    }
    object_map = {
        i: obj['#text']
        for i, obj
        in enumerate(question['objects']['object'])
    }

    return {
        'answers': None,
        'user_answer': [
            (object_map[int(match['@objectIndex'])], destination_map[int(match['@destinationIndex'])])
            for match in question['matches']['match']
        ],
        'correct_answer': [
            (object_map[int(match['@objectIndex'])], destination_map[int(match['@destinationIndex'])])
            for match in question['userAnswer']['match']
        ],
    }


def parse_essayQuestion(question):
    return {
        'answers': None,
        'user_answer':question['userAnswer'],
        'correct_answer': None,
    }


def parse_hotspotQuestion(question):
    return {
        'answers': None,
        'user_answer': question['userAnswer'],
        'correct_answer': question['hotspots'],
    }


parser_map = {
    'multipleResponseQuestion': parse_multipleResponseQuestion,
    'trueFalseQuestion': parse_trueFalseQuestion,
    'typeInQuestion': parse_typeInQuestion,
    'numericQuestion': parse_numericQuestion,
    'sequenceQuestion': parse_sequenceQuestion,
    'matchingQuestion': parse_matchingQuestion,
    'fillInTheBlankQuestion': parse_fillInTheBlankQuestion,
    'multipleChoiceTextQuestion': parse_multipleChoiceTextQuestion,
    'wordBankQuestion': parse_wordBankQuestion,
    'likertScaleQuestion': parse_likertScaleQuestion,
    'multipleChoiceQuestion': parse_multipleChoiceQuestion,
    'dndQuestion': parse_dndQuestion,
    'essayQuestion': parse_essayQuestion,
    'hotspotQuestion': parse_hotspotQuestion,
}


def parse_question(question_type, question):
    if isinstance(question, list):
        return (
            question_data
            for question_item in question
            for question_data in parse_question(question_type, question_item)
        )

    return (
        {
            **parser_map[question_type](question),
            **{
                'question_id': question['@id'],
                'question_type': question_type,
                'status': question['@status'],
                'evaluation_enabled': question['@evaluationEnabled'],
                'max_points': question.get('@maxPoints'),
                'max_attempts': question.get('@maxAttempts'),
                'awarded_points': question.get('@awardedPoints'),
                'used_attempts': question.get('usedAttempts'),
                'direction': question['direction']['text'],
                'feedback': question.get('feedback', {}).get('text'),
            }
        },
    )


def parse_ispring_data(ispring_suite_data_item):
    quiz_report = ispring_suite_data_item.get('dr', {}).get('quizReport')
    return {
        model_field: lookup_field(item=quiz_report, field=ispring_field)
        for ispring_field, model_field in ISPRING_REPORT_FIELDS_MAP.items()
    }
