from typing import Union

from kelvin.results.models import CourseLessonResult
from kelvin.scorm.models import Scorm, ScormResourceUserData


def update_scorm_points(scorm: Union[Scorm, ScormResourceUserData]) -> bool:
    if scorm.data:
        cmi = scorm.data.get('cmi', {})
    else:
        cmi = {}

    progress_score = extract_progress_score_from_cmi(cmi)
    score_updated = False
    if scorm.score_percent is None or progress_score['score_percent'] > scorm.score_percent:
        scorm.score_percent = progress_score['score_percent']
        score_updated = True
    if scorm.passed is None or progress_score['passed'] > scorm.passed:
        scorm.passed = progress_score['passed']
        score_updated = True

    return score_updated


def update_clesson_result_points(scorm: Union[Scorm, ScormResourceUserData]):
    clesson = scorm.clesson
    user = scorm.student
    passed = scorm.passed
    score_percent = scorm.score_percent

    clesson_result = (
        CourseLessonResult.objects
        .prefetch_related('summary')
        .filter(summary__clesson=clesson, summary__student=user)
        .first()
    )

    if not clesson_result:
        return

    if 0 <= score_percent <= 100:
        clesson_result.points = max(score_percent, clesson_result.points)

    if passed:
        clesson_result.answers = {
            '1': [
                {'answered': True}
            ]
        }

    clesson_result.save()


def module_passed(cmi):
    if cmi.get('core', {}).get('lesson_status') in ['passed', 'completed']:
        return True

    if cmi.get('completion_status') == 'completed':
        return True

    return False


def extract_progress_score_from_cmi(cmi):
    passed = module_passed(cmi)

    # Check `score` in root, else check in `core` key.
    if float(cmi.get('score', {}).get('raw', 0)) > 0:
        score = cmi['score']
    elif float(cmi.get('core', {}).get('score', {}).get('raw', 0)) > 0:
        score = cmi['core']['score']
    else:
        return {
            'passed': passed,
            'score_percent': 100 if passed else 0,
        }

    score_raw = float(score.get('raw', 0))
    score_max = float(score.get('max', 100))

    return {
        'passed': passed,
        'score_percent': int(round(score_raw / score_max * 100)) if score_max != 0 else 0,
    }
