import xmltodict

from django.shortcuts import get_object_or_404

from rest_framework import mixins, permissions, viewsets
from rest_framework.exceptions import ValidationError

from kelvin.common.db_functions import safe_get_or_create
from kelvin.courses.models import CourseLessonLink
from kelvin.scorm.models import Scorm, ScormResourceUserData
from kelvin.scorm.serializers import ScormISpringSuiteDataSerializer, ScormResourceUserDataSerializer, ScormSerializer


class ObjectScormDataAuthor(permissions.BasePermission):
    """
    Check if user is student that created scorm data
    """
    def has_object_permission(self, request, view, obj):
        return request.user.id == obj.student_id


class ScormViewSet(mixins.RetrieveModelMixin,
                   mixins.UpdateModelMixin,
                   viewsets.GenericViewSet):
    """
    Retrieve and update scorm user data in clesson
    """
    lookup_value_regex = '\d+'
    queryset = Scorm.objects.all()
    serializer_class = ScormSerializer
    permission_classes = [permissions.IsAuthenticated, ObjectScormDataAuthor]

    def get_object(self):
        """
        Get object by clesson_id and student_id.
        On PUT request get object from db or create it.
        """
        queryset = self.get_queryset()
        clesson = get_object_or_404(
            CourseLessonLink.objects.all(),
            pk=self.kwargs['pk'],
        )

        if self.request.method.lower() == 'put':
            get_method = safe_get_or_create
        else:
            get_method = get_object_or_404
        method_result = get_method(
            queryset,
            clesson=clesson,
            student=self.request.user,
        )
        if isinstance(method_result, tuple):
            obj, _ = method_result
        else:
            obj = method_result
        self.check_object_permissions(self.request, obj)
        return obj


class ScormResourceUserDataViewSet(
    mixins.RetrieveModelMixin,
    mixins.UpdateModelMixin,
    viewsets.GenericViewSet
):
    queryset = ScormResourceUserData.objects.all()
    serializer_class = ScormResourceUserDataSerializer
    permission_classes = [permissions.IsAuthenticated, ObjectScormDataAuthor]

    def get_object(self):
        """
        Get object by clesson_id and student_id.
        On PUT request get object from db or create it.
        """
        queryset = self.get_queryset()
        clesson = get_object_or_404(
            CourseLessonLink.objects.all(),
            pk=self.kwargs['scorm_package_pk'],
        )

        if self.request.method.lower() == 'put':
            get_method = safe_get_or_create
        else:
            get_method = get_object_or_404
        method_result = get_method(
            queryset,
            clesson=clesson,
            student=self.request.user,
            resource_ref=self.kwargs['pk'],
        )
        if isinstance(method_result, tuple):
            obj, _ = method_result
        else:
            obj = method_result
        self.check_object_permissions(self.request, obj)
        return obj


class ScormISpringSuiteViewSet(
    mixins.UpdateModelMixin,
    viewsets.GenericViewSet
):
    lookup_value_regex = '\d+'
    queryset = Scorm.objects.all()
    serializer_class = ScormISpringSuiteDataSerializer
    permission_classes = [permissions.IsAuthenticated, ObjectScormDataAuthor]

    def get_object(self):
        obj = get_object_or_404(
            self.get_queryset(),
            clesson_id=self.kwargs['pk'],
            student=self.request.user,
        )

        self.check_object_permissions(self.request, obj)
        return obj

    def perform_update(self, serializer):
        ispring_suite_data = serializer.instance.ispring_suite_data
        request_data = dict(self.request.data)
        try:
            request_data['dr'] = xmltodict.parse(request_data['dr'])
        except:
            pass
        qt = request_data.get('qt')
        if not qt:
            raise ValidationError('Parameter `qt` is not set')
        ispring_suite_data[qt] = request_data
        serializer.save(ispring_suite_data=ispring_suite_data)
