from builtins import map

from django.contrib import admin
from django.contrib.auth import get_user_model
from django.core.urlresolvers import reverse
from django.shortcuts import redirect, render, resolve_url
from django.utils.html import format_html, format_html_join
from django.utils.safestring import mark_safe

from kelvin.staff_notifications.admin.filters import CodeUsedFilter
from kelvin.staff_notifications.admin.forms import CodeBatchCreateForm
from kelvin.staff_notifications.models import ActivationCode, NewTeacherAction, NotificationEmail

User = get_user_model()


class NotificationEmailAdmin(admin.ModelAdmin):
    """
    Админка для модели `NotificationEmail`
    """
    list_display = ('id', 'email')


class NewTeacherActionAdmin(admin.ModelAdmin):
    """
    Админка для модели `NewTeacherAction`
    """
    list_display = (
        'id',
        'name',
    )


class ActivationCodeAdmin(admin.ModelAdmin):
    """
    Админка для модели `ActivationCode`
    """
    batch_create_template = 'staff_notifications/admin/batch_create.html'

    list_display = (
        'id',
        'issued',
        'user_id',
        'username',
        'code',
        'action_ids',
        'action_names',
    )
    list_display_links = (
        'code',
    )
    list_filter = (
        CodeUsedFilter,
    )
    search_fields = (
        'id',
        'activated_by__username',
        'code',
        'actions__id',
        'actions__name',
    )
    readonly_fields = (
        'code',
        'activated_by',
        'actions',
    )
    list_editable = (
        'issued',
    )

    CODE_ADMIN_LIST_REVERSE = 'admin:{0}_{1}_changelist'.format(
        ActivationCode._meta.app_label,
        ActivationCode._meta.model_name,
    )
    ACTION_ADMIN_CHANGE_REVERSE = 'admin:{0}_{1}_change'.format(
        NewTeacherAction._meta.app_label,
        NewTeacherAction._meta.model_name,
    )
    USER_ADMIN_CHANGE_REVERSE = 'admin:{0}_{1}_change'.format(
        User._meta.app_label,
        User._meta.model_name,
    )

    def get_queryset(self, request):
        """
        Сразу загружаем пользователя и действия, чтобы показать их
        в соответствующих полях
        """
        queryset = super(ActivationCodeAdmin, self).get_queryset(request)
        return (queryset.select_related('activated_by')
                        .prefetch_related('actions'))

    def user_id(self, obj):
        return obj.activated_by.id if obj.activated_by else ''
    user_id.short_description = u'ID пользователя'

    def username(self, obj):
        return format_html(
            u'<a href="{0}" target="_blank">{1}</a>',
            reverse(
                self.USER_ADMIN_CHANGE_REVERSE,
                args=(obj.activated_by.pk,)),
            obj.activated_by.username
        ) if obj.activated_by else ''
    username.short_description = u'Имя пользователя'

    def action_ids(self, obj):
        return format_html_join(
            mark_safe('<br/>'),
            '{}',
            ((action.pk,) for action in obj.actions.all())
        )
    action_ids.short_description = u'ID действий'

    def action_names(self, obj):
        return format_html_join(
            mark_safe('<br/>'),
            u'<a href="{0}" target="_blank">{1}</a>',
            ((reverse(
                self.ACTION_ADMIN_CHANGE_REVERSE,
                args=(action.pk,),
            ), action.name,) for action in obj.actions.all())
        )
    action_names.short_description = u'Названия действий'

    def add_view(self, request, form_url='', extra_context=None):
        """
        Генерируем набор кодов вместо создания одного кода
        """
        form = CodeBatchCreateForm(request.POST or None)
        if form.is_valid():
            count = form.cleaned_data['count']
            actions = form.cleaned_data['actions']
            created_ids = ActivationCode.batch_create(count, actions)
            return redirect(
                resolve_url(self.CODE_ADMIN_LIST_REVERSE) +
                '?id__in={}'.format(','.join(map(str, created_ids)))
            )

        opts = ActivationCode._meta
        context = {
            'title': u'Создание набора кодов',
            'add': True,
            'has_change_permission': self.has_change_permission(request),
            'opts': opts,
            'form': form,
        }
        context.update(extra_context or {})

        return render(request, self.batch_create_template, context)


admin.site.register(NotificationEmail, NotificationEmailAdmin)
admin.site.register(NewTeacherAction, NewTeacherActionAdmin)
admin.site.register(ActivationCode, ActivationCodeAdmin)
