from builtins import object, range
from collections import defaultdict

from django.conf import settings
from django.db import models, transaction
from django.db.models import Q
from django.utils import timezone
from django.utils.encoding import python_2_unicode_compatible
from django.utils.translation import ugettext_lazy as _

from kelvin.common.utils import generate_safe_code
from kelvin.courses.models import Course, CourseStudent
from kelvin.courses.services import copy_course
from kelvin.mail.controllers import SenderEmailController
from kelvin.staff_notifications.validators import validate_code
from kelvin.subjects.models import Subject


class NotificationEmailManager(models.Manager):
    """
    Менеджер рассылки оповещений о новых учителях
    """

    def notify_about_teacher(self, teacher, courses):
        """
        Посылает письма с сообщением о новом зарегистрированном
        учителе `teacher` всем записанным email
        """
        emails = self.get_queryset().values_list('email', flat=True)
        if not emails:
            return

        SenderEmailController.send_messages(
            receivers=emails,
            template_code=settings.TEACHER_NOTIFICATION_TEMPLATE_ID,
            user_id=teacher.id,
            name=teacher.get_full_name(),
            login=teacher.get_username(),
            courses=[{
                'course_id': course.id,
                'course_name': course.name,
                'course_code': course.code
            } for course in courses],
            admin_host=settings.ADMIN_HOST,
            frontend_host=settings.FRONTEND_HOST,
        )


class NotificationEmail(models.Model):
    """
    Модель для списка email для рассылки оповещений о новых учителях
    """
    email = models.EmailField(verbose_name=_('Email'), max_length=255, unique=True)

    objects = NotificationEmailManager()

    class Meta(object):
        verbose_name = _('Email рассылки о новых учителях')
        verbose_name_plural = _('Email рассылки о новых учителях')


@python_2_unicode_compatible
class NewTeacherAction(models.Model):
    """
    Модель для инициализации только что зарегистрировавшегося учителя
    """
    name = models.CharField(
        verbose_name=_('Название действия'),
        max_length=255,
    )
    subject = models.ForeignKey(
        Subject,
        verbose_name=_('Какой предмет проставить учителю'),
        blank=True,
        null=True,
    )
    course_to_own = models.ForeignKey(
        Course,
        verbose_name=_('Какой курс скопировать учителю'),
        related_name='+',
        blank=True,
        null=True,
    )
    course_to_add = models.ForeignKey(
        Course,
        verbose_name=_('В какой курс добавить как ученика'),
        related_name='+',
        blank=True,
        null=True,
    )
    open_lessons_count = models.IntegerField(
        verbose_name=_('Число открываемых занятий ученикам в скопированном'
                       ' курсе (проставляется дата назначения)'),
        default=1,
    )

    class Meta(object):
        verbose_name = _('Параметры действий при подтверждении учителя')
        verbose_name_plural = _(
            'Параметры действий при подтверждении учителя')

    def __str__(self):
        return self.name

    def do_with(self, teacher):
        """Выполняет действия с учителем"""

        messages = defaultdict(list)
        if teacher.is_teacher:
            messages['warning'].append('Учитель не требует подтверждения')
        else:
            teacher.is_teacher = True
            teacher.save()
            messages['info'].append('Учитель подтвержден')

        if self.subject:
            teacher.teacher_profile.subject_id = self.subject_id
            teacher.teacher_profile.save()
            messages['info'].append('Учителю проставлен предмет {0}'.format(
                self.subject.name))

        created_course = None
        if self.course_to_own:
            # ищем скопированный курс или создаем новый
            created_course = Course.objects.filter(
                Q(owner=teacher) &
                (Q(copy_of=self.course_to_own) | Q(pk=self.course_to_own.pk))
            ).first()
            if created_course:
                messages['warning'].append('Учитель уже имеет курс')
            else:
                created_course = Course.objects.get(id=self.course_to_own.id)
                copy_course(created_course, owner=teacher)
                messages['info'].append('Учителю скопирован курс: {0}'
                                        .format(self.course_to_own.name))

            # делаем занятия доступными учителю и выдаем первое занятие
            now = timezone.now()
            created_course.courselessonlink_set.update(
                accessible_to_teacher=now)
            if self.open_lessons_count < 0:
                created_course.courselessonlink_set.update(
                    date_assignment=now)
            else:
                for clesson in (
                        created_course.courselessonlink_set
                        .order_by('order')[:self.open_lessons_count]
                ):
                    clesson.date_assignment = now
                    clesson.save()

        if self.course_to_add:
            # Добавляем пользователя как ученика в курс
            course_student, created = CourseStudent.objects.get_or_create(
                student=teacher,
                course=self.course_to_add,
            )
            if not created:
                messages['info'].append('Учитель уже является учеником в этом курсе')

        return created_course, messages


@python_2_unicode_compatible
class ActivationCode(models.Model):
    """
    Модель для кодов активации
    """
    CODE_LENGTH = 10

    code = models.CharField(
        verbose_name=_('Код'),
        max_length=CODE_LENGTH,
        validators=[validate_code],
        unique=True,
    )
    activated_by = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        verbose_name=_('Пользователь, использовавший код'),
        null=True,
        blank=True,
    )
    actions = models.ManyToManyField(
        NewTeacherAction,
        verbose_name=_('Действия'),
    )
    issued = models.BooleanField(
        verbose_name=_('Код выдан учителю'),
        default=False,
    )

    class Meta(object):
        verbose_name = _('Код активации')
        verbose_name_plural = _('Коды активации')

    def __str__(self):
        return self.code

    @classmethod
    @transaction.atomic
    def batch_create(cls, count, actions):
        """
        Генерирует заданное количество кодов и привязывает их к действиям.
        Возвращает массив идентификаторов созданных кодов.
        """
        created_ids = []
        for _ in range(count):
            activation_code = ActivationCode(
                code=generate_safe_code(cls.CODE_LENGTH),
            )
            activation_code.save()
            activation_code.actions = actions
            created_ids.append(activation_code.pk)
        return created_ids

    @transaction.atomic
    def run_actions(self, user):
        """
        Запускает все действия привязанные к этому коду для переданного
        пользователя
        """
        for action in self.actions.all():
            action.do_with(user)
        self.activated_by = user
        self.save()
