from rest_framework import status, viewsets
from rest_framework.decorators import list_route
from rest_framework.generics import get_object_or_404
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response
from rest_framework.throttling import UserRateThrottle

from kelvin.staff_notifications.models import ActivationCode
from kelvin.staff_notifications.tasks import notify_about_teacher


class ActivationCodeViewSet(viewsets.GenericViewSet):
    """
    Работа с учительскими кодами
    """

    queryset = ActivationCode.objects.prefetch_related('actions')

    throttle_classes = [UserRateThrottle, ]
    permission_classes = [IsAuthenticated, ]

    @list_route(methods=['POST'])
    def activate(self, request):
        """
        Исполняет переданный код
        """

        code_object = get_object_or_404(
            self.queryset,
            code=request.data.get('activation_code', None),
        )

        if code_object.activated_by is None:
            code_object.run_actions(request.user)
            notify_about_teacher.delay(request.user.id)
        elif code_object.activated_by != request.user:
            return Response(status=status.HTTP_404_NOT_FOUND)

        return Response(status=status.HTTP_200_OK)
