import json

from future.builtins import range

from kelvin.courses.models import CourseLessonLink, CourseStudent
from kelvin.lessons.models import LessonProblemLink
from kelvin.results.models import CourseLessonResult, CourseLessonSummary
from kelvin.stress_fire import settings
from kelvin.stress_fire.ammo.base import Ammo
from kelvin.stress_fire.fixtures.answers import ANSWERS
from kelvin.stress_fire.fixtures.build_course import build_course
from kelvin.stress_fire.fixtures.users import create_students


class ClessonWebAmmo(Ammo):
    """
    Ручка /api/patch-clesson-results/
    """

    URL = '/api/v2/clessons/{course_lesson_id}/web/'
    METHOD = 'GET'
    TAG = 'SiriusCourses'

    def __init__(self, course_structure, user, query_params):
        super(ClessonWebAmmo, self).__init__()

        self.query_params = query_params

        self.course_structure = course_structure
        self.user = user

        self.course = self.course_structure['course_private']

        self.random_clesson = CourseLessonLink.objects.filter(
            course=self.course,
        ).order_by('?').first()

        self.random_problem_link = LessonProblemLink.objects.filter(
            lesson_id=self.random_clesson.lesson_id
        ).order_by('?').first()

    def get_url_params(self):
        return {
            'course_lesson_id': self.random_clesson.id
        }

    def get_headers(self, *args, **kwargs):
        return super().get_headers(*args, **kwargs) + [f'DEBUG-UID: {self.user.yauid}']


class GeneratorClessonWeb(object):
    PARAMS = {
        'count_users': None,
        'count_courses': None,
        'course_mode': None,
        'count_ammos': None,
        'count_problems': None,
        'is_evaluable': None,
    }

    def __init__(
        self,
        count_users=settings.DEFAULT_COUNT_OF_USERS,
        count_courses=settings.DEFAULT_COUNT_OF_COURSES,
        course_mode=settings.DEFAULT_COURSE_MODE,
        count_ammos=settings.DEFAULT_COUNT_OF_AMMOS,
        count_problems=settings.DEFAULT_COUNT_OF_PROBLEMS,
        count_lessons=settings.DEFAULT_COUNT_OF_LESSONS,
        expand_problems=False,
        expand_meta=False,
    ):
        self.expand_problems = expand_problems
        self.expand_meta = expand_meta
        self.count_ammos = count_ammos
        self.students_data = create_students(count_users)
        self.users = self.students_data['users']
        self.courses = [
            build_course(
                teacher=self.students_data['teacher'],
                course_mode=course_mode,
                count_problems=count_problems,
                count_lessons=count_lessons,
            ) for _ in range(count_courses)
        ]

        CourseStudent.objects.bulk_create([
            CourseStudent(
                course=course['course_private'],
                student=student,
            ) for student in self.users for course in self.courses
        ])

        self.count_users = len(self.users)
        self.count_courses = len(self.courses)

    def get_query_params(self):
        query_params = {}
        if self.expand_problems:
            query_params['expand_problems'] = True
        if self.expand_meta:
            query_params['expand_meta'] = True

        return query_params

    def generate_ammos(self):
        for ammo_number in range(self.count_ammos):
            yield ClessonWebAmmo(
                self.courses[ammo_number % self.count_courses],
                self.users[ammo_number % self.count_users],
                self.get_query_params(),
            ).build_request()
