import json
import uuid

from future.builtins import range

from kelvin.accounts.models import UserProject
from kelvin.courses.models import CourseLessonLink, CourseStudent
from kelvin.lessons.models import LessonProblemLink
from kelvin.projects.models import Project
from kelvin.results.models import CourseLessonResult, CourseLessonSummary
from kelvin.stress_fire import settings
from kelvin.stress_fire.ammo.base import Ammo
from kelvin.stress_fire.fixtures.answers import ANSWERS
from kelvin.stress_fire.fixtures.build_course import build_course
from kelvin.stress_fire.fixtures.users import create_students


class LibraryAmmo(Ammo):
    """
    Ручка /api/patch-clesson-results/
    """

    URL = '/api/v2/sirius-courses/library/'
    METHOD = 'GET'
    TAG = 'Library'

    def __init__(self, user):
        super(LibraryAmmo, self).__init__()
        self.user = user

    def get_headers(self, *args, **kwargs):
        return super().get_headers(*args, **kwargs) + [f'DEBUG-UID: {self.user.yauid}']


class GeneratorLibrary(object):
    PARAMS = {
        'count_users': None,
        'count_courses': None,
        'course_mode': None,
        'count_ammos': None,
        'count_problems': None,
        'is_evaluable': None,
    }

    def __init__(
        self,
        count_users=settings.DEFAULT_COUNT_OF_USERS,
        count_courses=20,
        course_mode=settings.DEFAULT_COURSE_MODE,
        count_ammos=settings.DEFAULT_COUNT_OF_AMMOS,
        count_problems=settings.DEFAULT_COUNT_OF_PROBLEMS,
        count_lessons=settings.DEFAULT_COUNT_OF_LESSONS,
    ):
        self.count_ammos = count_ammos
        self.students_data = create_students(count_users)
        self.users = self.students_data['users']
        self.project = Project.objects.create(
            slug=str(uuid.uuid4()),
            title=str(uuid.uuid4()),
            description=str(uuid.uuid4()),
        )
        self.courses = [
            build_course(
                teacher=self.students_data['teacher'],
                project=self.project,
                course_mode=course_mode,
                count_problems=count_problems,
                count_lessons=count_lessons,
            ) for _ in range(count_courses)
        ]

        self.deleted_courses = [
            build_course(
                teacher=self.students_data['teacher'],
                project=self.project,
                course_mode=course_mode,
                count_problems=count_problems,
                count_lessons=count_lessons,
            ) for _ in range(count_courses)
        ]

        CourseStudent.objects.bulk_create(
            [
                CourseStudent(
                    course=course['course_private'],
                    student=student,
                    deleted=False,
                ) for student in self.users for course in self.courses
            ] + [
                CourseStudent(
                    course=course['course_private'],
                    student=student,
                    deleted=True,
                ) for student in self.users for course in self.deleted_courses
            ]
        )

        UserProject.objects.bulk_create(
            [
                UserProject(
                    user=user, project=self.project
                ) for user in self.users
            ]
        )

        self.count_users = len(self.users)
        self.count_courses = len(self.courses)

    def generate_ammos(self):
        for ammo_number in range(self.count_ammos):
            yield LibraryAmmo(
                self.users[ammo_number % self.count_users],
            ).build_request()
