import json
import uuid

from future.builtins import range

from kelvin.accounts.models import UserProject
from kelvin.courses.models import CourseLessonLink, CourseStudent
from kelvin.lessons.models import LessonProblemLink
from kelvin.projects.models import Project
from kelvin.results.models import CourseLessonResult, CourseLessonSummary
from kelvin.stress_fire import settings
from kelvin.stress_fire.ammo.base import Ammo
from kelvin.stress_fire.fixtures.answers import ANSWERS
from kelvin.stress_fire.fixtures.build_course import build_course
from kelvin.stress_fire.fixtures.users import create_students


class CourseStrictAmmo(Ammo):
    """
    Ручка /api/patch-clesson-results/
    """

    URL = '/api/v2/sirius-courses/{course_id}/strict/'
    METHOD = 'GET'
    TAG = 'CourseStrict'

    def __init__(
        self, course_structure, user, answer, problem_number,
        is_evaluable
    ):
        super(CourseStrictAmmo, self).__init__()

        self.is_evaluable = is_evaluable
        self.problem_number = problem_number
        self.course_structure = course_structure
        self.user = user
        self.answer = answer

        self.course = self.course_structure['course_private']

        self.random_clesson = CourseLessonLink.objects.filter(
            course=self.course,
        ).order_by('?').first()

        self.random_problem_link = LessonProblemLink.objects.filter(
            lesson_id=self.random_clesson.lesson_id
        ).order_by('?').first()

        summary, created = CourseLessonSummary.objects.get_or_create(
            student=self.user,
            clesson=self.random_clesson,
        )
        if created:
            self.result = CourseLessonResult.objects.create(
                summary=summary,
                answers={},
                points=0,
                max_points=300,
                completed=False,
            )

        else:
            self.result = summary.courselessonresult_set.first()

    def get_url_params(self):
        return {
            'course_id': self.course.id
        }

    def get_headers(self, *args, **kwargs):
        return super().get_headers(*args, **kwargs) + [f'DEBUG-UID: {self.user.yauid}']

    def get_body(self):
        return {}


class GeneratorCourseStrictAmmo(object):
    """
    Ручка /api/patch-clesson-results/
    Ученики по очереди шлют запросы в курсы.
    Все студенты являются учениками всех курсов.
    Например: учеников 3, курсов 2.
    Патроны (ученик - курс):
    0 - 0
    1 - 1
    2 - 0
    3 - 1
    ...
    """
    PARAMS = {
        'count_users': None,
        'count_courses': None,
        'course_mode': None,
        'count_ammos': None,
        'count_problems': None,
        'is_evaluable': None,
    }

    def __init__(
        self,
        count_users=settings.DEFAULT_COUNT_OF_USERS,
        count_courses=settings.DEFAULT_COUNT_OF_COURSES,
        course_mode=settings.DEFAULT_COURSE_MODE,
        count_ammos=settings.DEFAULT_COUNT_OF_AMMOS,
        count_problems=settings.DEFAULT_COUNT_OF_PROBLEMS,
        count_lessons=settings.DEFAULT_COUNT_OF_LESSONS,
        is_evaluable=True,
        answers=ANSWERS,
    ):
        self.is_evaluable = is_evaluable
        self.count_ammos = count_ammos
        self.students_data = create_students(count_users)
        self.users = self.students_data['users']
        self.project = Project.objects.create(
            slug=str(uuid.uuid4()),
            title=str(uuid.uuid4()),
            description=str(uuid.uuid4()),
        )
        self.courses = [
            build_course(
                teacher=self.students_data['teacher'],
                project=self.project,
                course_mode=course_mode,
                count_problems=count_problems,
                count_lessons=count_lessons,
            ) for _ in range(count_courses)
        ]
        self.answers = answers

        CourseStudent.objects.bulk_create([
            CourseStudent(
                course=course['course_private'],
                student=student,
            ) for student in self.users for course in self.courses
        ])

        UserProject.objects.bulk_create(
            [
                UserProject(
                    user=user, project=self.project
                ) for user in self.users
            ]
        )

        self.count_users = len(self.users)
        self.count_courses = len(self.courses)
        self.count_answers = len(self.answers)

    def generate_ammos(self):
        for ammo_number in range(self.count_ammos):
            yield CourseStrictAmmo(
                self.courses[ammo_number % self.count_courses],
                self.users[ammo_number % self.count_users],
                self.answers[
                    (ammo_number // self.count_users) % self.count_answers
                ],
                (ammo_number // self.count_users) // self.count_answers,
                self.is_evaluable,
            ).build_request()
