# coding=utf-8
import time

import requests
from yql.api.v1.client import YqlClient

from django.core.files.base import ContentFile
from django.core.files.storage import default_storage

from kelvin.stress_fire.settings import OAUTH_TOKEN, SANDBOX_HOST, SANDBOX_SECRET, STATUS_SUCCESS, STATUSES_FAIL

QUERY = """
$script = @@
import re

match = re.compile('Session_id_safe=.*?\|(\d+)\.')

def parse_uid(cookie):
    result = match.search(cookie)
    return result.group(1) if result else ''
@@;

$callable = Python::parse_uid("(String?)->String", $script);

USE hahn;
SELECT
    $callable(cookies)
FROM
    hahn.[home/logfeller/logs/qloud-router-log/1d/{date}]
WHERE
    $callable(cookies) != ''
LIMIT {lines};
"""


def get_ya_user_ids(lines, date_string):
    """
    Gets cookies from YT and parse them to extract yandex user_ids.
    Return user_ids in one string separated by new line.
    """

    client = YqlClient(db='hahn')
    request = client.query(QUERY.format(lines=lines, date=date_string))
    request.run()

    uids = set()

    for table in request.get_results():
        for row in table.rows:
            uids.add(row[0])

    return '\n'.join(uids)


def authenticate(session):
    """Add oauth token to session`s headers."""

    if not OAUTH_TOKEN:
        raise Exception('OAUTH_TOKEN not exist')

    session.headers.update({'Authorization': 'OAuth {}'.format(OAUTH_TOKEN)})


def run_new_task(session, uids):
    custom_fields = {'uids': uids}
    custom_fields.update(SANDBOX_SECRET)

    task_data = {
        'type': 'GENERATE_STRESS_SESSION_IDS',
        'custom_fields': [
            {'name': k, 'value': v} for k, v in custom_fields.items()
        ],
        'description': 'Just generate stress session ids',
        'notifications': [{
            'recipients': [],
            'statuses': [],
            'transport': 'jabber',
        }],
        'owner': 'EDUCATION',
        'priority': {
            'class': 'USER',
            'subclass': 'NORMAL',
        },
    }
    result = session.post(
        '{host}/task'.format(host=SANDBOX_HOST),
        json=task_data,
    )
    task_id = result.json().get('id')

    result = session.put(
        '{host}/batch/tasks/start'.format(host=SANDBOX_HOST),
        json={'id': [task_id]},
    )

    if result.status_code != 200:
        raise Exception('Task not created')

    return task_id


def get_executed_resource(session, task_id):
    status = 'DRAFT'
    while status != STATUS_SUCCESS:
        old_status = status
        result = session.get('{host}/task/{task_id}'.format(
            host=SANDBOX_HOST,
            task_id=task_id,
        ))
        if result.status_code != 200:
            time.sleep(10)
            continue

        status = result.json().get('status')

        if status == old_status:
            time.sleep(10)
            continue

        if status in STATUSES_FAIL:
            raise Exception('Error with sandbox task execution')

    result = session.get('{host}/task/{task_id}/resources'.format(
        host=SANDBOX_HOST,
        task_id=task_id,
    ))
    resources = result.json()['items']
    for resource in resources:
        if resource['type'] == 'STRESS_SESSION_IDS':
            return resource


def save_resource(session, url, file_name):
    """Save session_ids from sandbox to local file."""

    content = ContentFile(session.get(url).text)
    default_storage.save(file_name, content)


def generate_session_ids(file_name, date, lines):
    """
    Generate session_ids for stress tests and save it to the file.
    """

    # Create auth session
    session = requests.session()
    authenticate(session)

    # Create and run task
    user_ids = get_ya_user_ids(lines, date)
    task_id = run_new_task(session, user_ids)

    # Wait sandbox executing
    resource = get_executed_resource(session, task_id)

    # Save result in file
    save_resource(session, resource['http']['proxy'], file_name)


def get_sessions(file_name, date, lines):
    """
    Open cached sessions file.
    If file not exist or outdated - will generate new.
    """

    if not default_storage.exists(file_name):
        generate_session_ids(file_name, date, lines)

    return default_storage.open(file_name).readlines()
