import logging
import re
import sys
from builtins import object, str
from datetime import datetime

from dateutil.relativedelta import *

from django.conf import settings
from django.contrib.auth import get_user_model
from django.contrib.contenttypes.fields import GenericForeignKey
from django.contrib.contenttypes.models import ContentType
from django.contrib.postgres.fields import JSONField
from django.db import models
from django.utils.encoding import python_2_unicode_compatible
from django.utils.translation import ugettext_lazy as _

from kelvin.common.staff_reader import staff_reader

User = get_user_model()

logger = logging.getLogger(__name__)


class TagTypeBase(object):
    ALLOWED_OPERATIONS = ["==", "!=", ">", "<", ">=", "<=", "regexp", ]
    OPERATIONS = []

    def __init__(self):
        # множество OPERATIONS должно быть подмножеством ALLOWED_OPERATIONS
        if set(self.OPERATIONS) > set(self.ALLOWED_OPERATIONS):
            raise RuntimeError("Your operations is not subset of allowed ones")

        if set(self.OPERATION_FUNCS.keys()) > set(self.ALLOWED_OPERATIONS):
            raise RuntimeError("Map of operations contains something unallowed")

    @staticmethod
    def is_dynamic():
        return False

    @staticmethod
    def visible():
        return False

    @staticmethod
    def get_source_values_for_object(object):
        """
            Метод для получения значения тега из источника.
            У каждого типа тега источник может быть свой.
            Метод абстрагирует источник данных.
            :return: string или результат вызова casting_callback над найденным значением
        """
        raise NotImplementedError()

    @staticmethod
    def get_semantic_type_name():
        """
        Метод для получения семантического названия типа
        :return: string
        """
        raise NotImplementedError()

    @staticmethod
    def get_semantic_type_name_l10n():
        """
        Метод для получения семантического названия типа
        :return: string
        """
        raise NotImplementedError

    @staticmethod
    def get_language_type_name():
        """
        Метод для получения языкового названия типа
        :return: string
        """
        raise NotImplementedError()

    @classmethod
    def get_operations(cls):
        """
        Метод для получения списка поддерживаемых операций,
        где каждая операция представлена строкой ( например ">", "<" )
        :return: [ <op1>, <op2>, ..., <opN> ]
        """
        return cls.OPERATIONS

    @staticmethod
    def get_db_type():
        raise NotImplementedError()

    @staticmethod
    def cast(object):
        return object

    @classmethod
    def is_valid(cls, value):
        try:
            return cls.cast(value) is not None
        except:
            return False

    @classmethod
    def modify_suggest_query_param(cls, query_string):
        return query_string

    @staticmethod
    def __eq(object_values, target_value):
        """ == """
        for value in object_values:
            if value == target_value:
                return True
        return False

    @staticmethod
    def __neq(object_values, target_value):
        """ != """
        return not TagTypeBase.__eq(object_values, target_value)

    @staticmethod
    def __re(object_values, target_value):
        """ regexp match """
        regexp = re.compile(target_value)
        for value in object_values:
            if regexp.match(value) is not None:
                return True
        return False

    @staticmethod
    def __gt(object_values, target_value):
        """ > """
        for value in object_values:
            if value > target_value:
                return True
        return False

    @staticmethod
    def __gte(object_values, target_value):
        """ >= """
        for value in object_values:
            if value >= target_value:
                return True
        return False

    @staticmethod
    def __lt(object_values, target_value):
        """ < """
        for value in object_values:
            if value < target_value:
                return True
        return False

    @staticmethod
    def __lte(object_values, target_value):
        """ <= """
        for value in object_values:
            if value <= target_value:
                return True
        return False

    @staticmethod
    def __range(object_values, target_value):
        """
        :param object_values: список с одной датой - датой выхода на работу в Яндекс
        :param target_value: список с двумя датами - начало и конец диапазона ( из значения предиката КНФ )
        :return: True/False
        """
        try:
            return object_values[0] >= target_value[0] and object_values[0] <= target_value[1]
        except Exception as e:
            raise RuntimeError("Something strange: {}".format(str(e)))

    OPERATION_FUNCS = {
        "==": __eq.__func__,
        "!=": __neq.__func__,
        ">": __gt.__func__,
        ">=": __gte.__func__,
        "<": __lt.__func__,
        "<=": __lte.__func__,
        "regexp": __re.__func__,
        "range": __range.__func__,
    }

    @classmethod
    def get_operation_funcs(cls):
        return cls.OPERATION_FUNCS

    @classmethod
    def perform_boolean_operation(cls, operation, target_value, object):
        """ Метод для выполнения булевой операции над тегом. """
        if operation not in cls.get_operations():
            raise RuntimeError("Attempt to perform unsupported operation {}".format(operation))

        if operation not in list(cls.OPERATION_FUNCS.keys()):
            raise RuntimeError("Attempt to use operation '{}' which is not mapped into any function".format(operation))

        source_values = cls.get_source_values_for_object(object)
        result = cls.get_operation_funcs()[operation](
            [cls.cast(x) for x in source_values],
            cls.cast(target_value)
        )
        return result


class TagTypeProject(TagTypeBase):
    """
        Простой строковый тип тега, для возможности произвольного тегирования пользователей администратором проекта.
        Поддерживаются операции: ==, !=
        Это статический тип тега, то есть предполагается поиск его значения в таблице Tags
    """

    OPERATIONS = ["==", "!="]

    @staticmethod
    def is_dynamic():
        return False

    @staticmethod
    def get_semantic_type_name():
        return "Project"

    @staticmethod
    def get_semantic_type_name_l10n():
        return "Проект"

    @staticmethod
    def get_language_type_name():
        return "integer"

    @staticmethod
    def get_db_type():
        return "PRJ"

    @staticmethod
    def cast(value):
        try:
            return int(value)
        except:
            raise RuntimeError("Cannot convert value {} to int".format(value))

    @staticmethod
    def get_source_values_for_object(object):
        """
        Для переданного объекта ищем все теги с типом Project.
        :return: возвращаем массив, так как тегов такого типа может быть много
        """
        django_model = object._meta.model
        tag_type = TagTypeProject.get_db_type()
        if isinstance(object, User):
            return [int(i) for i in object.tag_values.get(tag_type, [])]

        tagged_object_items = TaggedObject.objects.filter(
            object_id=object.id,
            content_type=ContentType.objects.get_for_model(django_model),
            tag__type=tag_type,
        ).select_related('tag')
        return [int(x.tag.value) for x in tagged_object_items]


class TagTypeUser(TagTypeBase):
    """
        Простой числовой тип тега
        Поддерживаются операции: ==, !=
        Это статический тип тега, то есть предполагается поиск его значения в таблице Tags
    """

    OPERATIONS = ["==", "!="]

    @staticmethod
    def is_dynamic():
        return False

    @staticmethod
    def cast(value):
        try:
            return int(value)
        except:
            raise RuntimeError("Cannot convert value {} to int".format(value))

    @staticmethod
    def get_semantic_type_name():
        return "User"

    @staticmethod
    def get_semantic_type_name_l10n():
        return "Пользователь"

    @staticmethod
    def get_language_type_name():
        return "integer"

    @staticmethod
    def get_db_type():
        return "USR"

    @staticmethod
    def get_source_values_for_object(object):
        """
        Для переданного объекта ищем все теги с типом User.
        :return: возвращаем массив, так как тегов такого типа может быть много
        """
        django_model = object._meta.model
        tag_type = TagTypeUser.get_db_type()
        if isinstance(object, User):
            return [int(i) for i in object.tag_values.get(tag_type, [])]

        tagged_object_items = TaggedObject.objects.filter(
            object_id=object.id,
            content_type=ContentType.objects.get_for_model(django_model),
            tag__type=tag_type,
        ).select_related('tag')

        return [int(x.tag.value) for x in tagged_object_items]


class TagTypeStaffGroup(TagTypeBase):
    """
        Статический тип тега с именем группы.
        Значением является идентификатор группы на Staff-е.
        Значение берется из таблицы Tag, данные в которую заливаются периодчески celery-таской
    """

    OPERATIONS = ["==", "!="]

    @staticmethod
    def is_dynamic():
        return False

    @classmethod
    def is_valid(cls, value):
        if not value:
            return False
        return True

    @staticmethod
    def visible():
        return True

    @staticmethod
    def get_semantic_type_name():
        return "StaffGroup"

    @staticmethod
    def get_semantic_type_name_l10n():
        return "Группа на стаффе"

    @staticmethod
    def get_language_type_name():
        return "string"

    @staticmethod
    def get_db_type():
        return "STGRP"

    @staticmethod
    def get_source_values_for_object(object):
        """
            Для переданного объекта ищем все теги с типом StaffGroup.
            :return: возвращаем массив, так как тегов такого типа может быть много
        """

        django_model = object._meta.model
        tag_type = TagTypeStaffGroup.get_db_type()
        if isinstance(object, User):
            return object.tag_values.get(tag_type, [])

        tagged_object_items = TaggedObject.objects.filter(
            object_id=object.id,
            content_type=ContentType.objects.get_for_model(django_model),
            tag__type=tag_type,
        ).select_related('tag')

        return [x.tag.value for x in tagged_object_items]


class TagTypeStaffStartDate(TagTypeBase):
    """
        Динамический тип тега с датой выхода человека на работу ( в терминах Staff ).
        Значение берется из микросервиса staff_reaader через соответствующий kelvin-интерфейс
    """

    OPERATIONS = ["==", "!=", "<", ">", "<=", ">=", "range"]

    @staticmethod
    def is_dynamic():
        return True

    @staticmethod
    def visible():
        return True

    @staticmethod
    def get_semantic_type_name():
        return "StaffStartDate"

    @staticmethod
    def get_semantic_type_name_l10n():
        return "Дата выхода на работу"

    @staticmethod
    def get_language_type_name():
        return "date"

    @staticmethod
    def get_db_type():
        return "STDT"

    @classmethod
    def get_staff_start_date(cls, obj):
        """
            Идем в staff_reader  в ручку suggest_user и получаем дату начала работы ( YYYY-MM-DD )
        """
        try:
            return staff_reader.get_suggestuser(obj.username)["official"]["join_at"]
        except KeyError as e:
            logger.error("Something strange in staff user data: {}" . format(str(e)))
            return None

    @staticmethod
    def get_source_values_for_object(obj):
        start_date = TagTypeStaffStartDate.get_staff_start_date(obj)
        return [start_date] if start_date else []

    @staticmethod
    def cast(obj):
        """
            Переопределяем метод базового класса для превращения строки в объект 'datetime',
            на котором уже будут определены наши OPERATIONS
            Иногда ( например в случае операции range ) в obj может быть строка вида:
                %Y-%m-%d;%Y-%m-%d - диапазон дат через ';'
            - в этом случае наш кастинг должен вернуть список дат из двух элементов
        """
        try:
            if re.match(r"^\d{4}-\d{2}-\d{2};\d{4}-\d{2}-\d{2}$", obj):
                dates = obj.split(';')
                return [datetime.strptime(x, "%Y-%m-%d") for x in dates]
            else:
                return datetime.strptime(obj, "%Y-%m-%d")
        except TypeError as e:
            # TODO: sentry
            raise RuntimeError("Casting object it not type of str: {}".format(str(e)))
        except ValueError as e:
            # TODO: sentry
            raise RuntimeError("String {} considered to be a date in format \%Y-\%m-\%d".format(str(obj)))


class TagTypeStaffTimeFromStartDate(TagTypeStaffStartDate):
    """
        Динамический тип тега "колчество дней с момента выхода на работу"
        Значение берется из микросервиса staff_reaader через соответствующий kelvin-интерфейс
    """
    OPERATIONS = ["==", "!=", "<", ">", "<=", ">="]

    @staticmethod
    def visible():
        return True

    @staticmethod
    def get_semantic_type_name():
        return "StaffTimeFromStartDate"

    @staticmethod
    def get_semantic_type_name_l10n():
        return "Количество дней с момента выхода на работу"

    @staticmethod
    def get_language_type_name():
        return "integer"

    @staticmethod
    def get_db_type():
        return "STFSD"

    @staticmethod
    def get_source_values_for_object(object):
        """
        Идем в staff_reader  в ручку suggest_user и получаем дату начала работы ( YYYY-MM-DD )
        Затем считаем, сколько прошло дней с момента выхода на работу.
        """
        staff_start_date = datetime.strptime(
            TagTypeStaffTimeFromStartDate.get_staff_start_date(object),
            "%Y-%m-%d"
        )
        return [(datetime.today() - staff_start_date).days]

    @staticmethod
    def cast(obj):
        """
        Если obj - просто целое число, то считаем, что это количвество дней.
        Если obj вида r'^\d+[ymd]$', то значит, это значение из формулы и надо привести к дням с текущего момента
        """
        def __dimension_to_days(value, dimension):
            now = datetime.now()
            if dimension == "d":
                return int(value)
            if dimension == "m":
                return (now - (now - relativedelta(months=value))).days
            if dimension == "y":
                return (now - (now - relativedelta(years=value))).days

        if re.match(r"^\d+$", str(obj)):
            return int(obj)
        m = re.match(r"^(\d+)([ymd])$", str(obj))
        if m:
            return __dimension_to_days(
                value=int(m.group(1)),
                dimension=m.group(2),
            )

        raise RuntimeError("String with time delta is in unknown format: {}".format(str(obj)))


class TagTypeIsChief(TagTypeBase):
    """
    Динамический тип тега "Является ли человек руководителем"
    Значение берется из микросервиса staff_reaader через соответствующий kelvin-интерфейс
    """
    OPERATIONS = ["=="]

    @staticmethod
    def visible():
        return True

    @staticmethod
    def is_dynamic():
        return True

    @staticmethod
    def get_semantic_type_name():
        return "IsChief"

    @staticmethod
    def get_semantic_type_name_l10n():
        return "Является ли сотрудник руководителем"

    @staticmethod
    def get_language_type_name():
        return "bool"

    @staticmethod
    def get_db_type():
        return "ISCHF"

    @staticmethod
    def get_source_values_for_object(obj):
        """
        Идем в staff_reader  в ручку suggest_user и получаем дату начала работы ( YYYY-MM-DD )
        Затем считаем, сколько прошло дней с момента выхода на работу.
        """
        start_date = staff_reader.get_chief_start_date(
            username=obj.username
        )
        if not start_date:
            return [0]

        return [1]

    @staticmethod
    def cast(obj):
        """
        Переопределяем метод базового класса для превращения строки в объект 'bool',
        на котором уже будут определены наши OPERATIONS
        """
        if obj in set(["1", 1]):
            return True
        if obj in set(["0", 0]):
            return False
        raise RuntimeError("Unpredicatble string representation for boolean type: {}".format(str(object)))


class TagTypeChiefDuration(TagTypeBase):
    """
    Динамический тип тега "Как долго человек является руководителем к текущему моменту"
    Значение берется из микросервиса staff_reaader через соответствующий kelvin-интерфейс
    """
    OPERATIONS = ["==", "!=", "<", ">", "<=", ">="]

    @staticmethod
    def visible():
        return True

    @staticmethod
    def is_dynamic():
        return True

    @staticmethod
    def get_semantic_type_name():
        return "ChiefDuration"

    @staticmethod
    def get_semantic_type_name_l10n():
        return "Продолжительность работы руководителем"

    @staticmethod
    def get_language_type_name():
        return "integer"

    @staticmethod
    def get_db_type():
        return "CHFDRTN"

    @staticmethod
    def get_source_values_for_object(obj):
        """
        Идем в staff_reader  в ручку suggest_user и получаем дату начала работы ( YYYY-MM-DD )
        Затем считаем, сколько прошло дней с момента выхода на работу.
        """
        start_date = staff_reader.get_chief_start_date(
            username=obj.username
        )
        if not start_date:
            return [0]

        start_date_dt = datetime.strptime(
            start_date,
            "%Y-%m-%dT%H:%M:%S"
        )
        return [(datetime.now() - start_date_dt).days]

    @staticmethod
    def cast(obj):
        """
        Если obj - просто целое число, то считаем, что это количвество дней.
        Если obj вида r'^\d+[ymd]$', то значит, это значение из формулы и надо привести к дням с текущего момента
        """
        def __dimension_to_days(value, dimension):
            now = datetime.now()
            if dimension == "d":
                return int(value)
            if dimension == "m":
                return (now - (now - relativedelta(months=value))).days
            if dimension == "y":
                return (now - (now - relativedelta(years=value))).days

        if re.match(r"^\d+$", str(obj)):
            return int(obj)
        m = re.match(r"^(\d+)([ymd])$", str(obj))
        if m:
            return __dimension_to_days(
                value=int(m.group(1)),
                dimension=m.group(2),
            )

        raise RuntimeError("String with time delta is in unknown format: {}".format(str(obj)))


class TagTypeStaffOffice(TagTypeBase):
    """
        Статический тип тега с именем офиса.
        Значением является идентификатор офиса на Staff-е.
        Значение берется из таблицы Tag, данные в которую заливаются периодчески celery-таской
    """

    OPERATIONS = ["==", "!="]

    @staticmethod
    def is_dynamic():
        return False

    @classmethod
    def is_valid(cls, value):
        if not value:
            return False
        return True

    @staticmethod
    def visible():
        return True

    @staticmethod
    def get_semantic_type_name():
        return "StaffOffice"

    @staticmethod
    def get_semantic_type_name_l10n():
        return "Офис на стаффе"

    @staticmethod
    def get_language_type_name():
        return "string"

    @staticmethod
    def get_db_type():
        return "STOFF"

    @staticmethod
    def get_source_values_for_object(object):
        """
            Для переданного объекта ищем все теги с типом StaffOffice.
            :return: возвращаем массив, так как тегов такого типа может быть много
        """

        django_model = object._meta.model
        tag_type = TagTypeStaffOffice.get_db_type()
        if isinstance(object, User):
            return object.tag_values.get(tag_type, [])

        tagged_object_items = TaggedObject.objects.filter(
            object_id=object.id,
            content_type=ContentType.objects.get_for_model(django_model),
            tag__type=tag_type,
        )
        return [x.tag.value for x in tagged_object_items]


class TagTypeStaffCity(TagTypeBase):
    """
        Статический тип тега с именем города.
        Значением является идентификатор города на Staff-е.
        Значение берется из таблицы Tag, данные в которую заливаются периодчески celery-таской
    """

    OPERATIONS = ["==", "!="]

    @staticmethod
    def is_dynamic():
        return False

    @classmethod
    def is_valid(cls, value):
        if not value:
            return False
        return True

    @staticmethod
    def visible():
        return True

    @staticmethod
    def get_semantic_type_name():
        return "StaffCity"

    @staticmethod
    def get_semantic_type_name_l10n():
        return "Город на стаффе"

    @staticmethod
    def get_language_type_name():
        return "string"

    @staticmethod
    def get_db_type():
        return "STCTY"

    @staticmethod
    def get_source_values_for_object(object):
        """
            Для переданного объекта ищем все теги с типом StaffCity.
            :return: возвращаем массив, так как тегов такого типа может быть много
        """

        django_model = object._meta.model
        tag_type = TagTypeStaffCity.get_db_type()
        if isinstance(object, User):
            return object.tag_values.get(tag_type, [])

        tagged_object_items = TaggedObject.objects.filter(
            object_id=object.id,
            content_type=ContentType.objects.get_for_model(django_model),
            tag__type=tag_type,
        ).select_related('tag')
        return [x.tag.value for x in tagged_object_items]


class TagTypeCourse(TagTypeBase):
    """
    Простой строковый тип тега, для возможности произвольного тегирования пользователей администратором проекта.
    Поддерживаются операции: ==, !=
    Это статический тип тега, то есть предполагается поиск его значения в таблице Tags
    """

    OPERATIONS = ["==", "!="]

    @staticmethod
    def is_dynamic():
        return False

    @staticmethod
    def cast(value):
        try:
            int(value)
        except:
            raise RuntimeError("Cannot convert value {} to int".format(value))

    @staticmethod
    def get_semantic_type_name():
        return "Course"

    @staticmethod
    def get_semantic_type_name_l10n():
        return "Курсы"

    @staticmethod
    def get_language_type_name():
        return "integer"

    @staticmethod
    def get_db_type():
        return "CRS"

    # TODO: too much code repeatance with TagTypeProject
    @staticmethod
    def get_source_values_for_object(object):
        """
        Для переданного объекта ищем все теги с типом Project.
        :return: возвращаем массив, так как тегов такого типа может быть много
        """
        django_model = object._meta.model
        tag_type = TagTypeCourse.get_db_type()
        if isinstance(object, User):
            return [int(i) for i in object.tag_values.get(tag_type, [])]

        tagged_object_items = TaggedObject.objects.filter(
            object_id=object.id,
            content_type=ContentType.objects.get_for_model(django_model),
            tag__type=tag_type,
        ).select_related('tag')
        return [int(x.tag.value) for x in tagged_object_items]


class TagTypeUserBase(TagTypeBase):
    """
    Простой строковый тип тега, для возможности произвольного тегирования пользователей администратором проекта.
    Поддерживаются операции: ==, !=
    Это статический тип тега, то есть предполагается поиск его значения в таблице Tags
    """

    OPERATIONS = ["==", "!="]

    @staticmethod
    def is_dynamic():
        return False

    @staticmethod
    def cast(value):
        try:
            int(value)
        except:
            raise RuntimeError("Cannot convert value {} to int".format(value))

    @staticmethod
    def get_language_type_name():
        return "integer"

    @staticmethod
    def get_source_values_for_object(object):
        """
        Для переданного объекта ищем все теги с типом Project.
        :return: возвращаем массив, так как тегов такого типа может быть много
        """
        django_model = object._meta.model
        tag_type = TagTypeUserBase.get_db_type()
        if isinstance(object, User):
            return [int(i) for i in object.tag_values.get(tag_type, [])]

        tagged_object_items = TaggedObject.objects.filter(
            object_id=object.id,
            content_type=ContentType.objects.get_for_model(django_model),
            tag__type=tag_type,
        ).select_related('tag')
        return [int(x.tag.value) for x in tagged_object_items]


class TagTypeStaffHRBP(TagTypeUserBase):
    @staticmethod
    def get_db_type():
        return "STHRBP"

    @staticmethod
    def get_semantic_type_name():
        return "StaffHRBP"

    @staticmethod
    def get_semantic_type_name_l10n():
        return "HR-партнер на стаффе"


class TagTypeStaffChief(TagTypeUserBase):
    @staticmethod
    def get_db_type():
        return "STCHF"

    @staticmethod
    def get_semantic_type_name():
        return "StaffChief"

    @staticmethod
    def get_semantic_type_name_l10n():
        return "Руководитель на стаффе"


class TagTypeGenericString(TagTypeBase):
    """
        Строковый тип тега с произвольным смыслом, который в него может заложить пользователь
        Поддерживает операции ==, !=, regexp
        Это статический тип тега, то есть предполагается поиск его значения в таблице Tags
    """

    OPERATIONS = ["==", "!=", "regexp"]

    @staticmethod
    def is_dynamic():
        return False

    @staticmethod
    def get_semantic_type_name():
        return "GenericString"

    @staticmethod
    def get_semantic_type_name_l10n():
        return "Произвольная строка"

    @staticmethod
    def get_language_type_name():
        return "string"

    @staticmethod
    def get_db_type():
        return "GNRSTR"

    @staticmethod
    def get_source_values_for_object(object):
        """
        Для переданного объекта ищем все теги с типом GenericString.
        :return: возвращаем массив, так как тегов такого типа может быть много
        """
        django_model = object._meta.model
        tag_type = TagTypeGenericString.get_db_type()
        if isinstance(object, User):
            return object.tag_values.get(tag_type, [])

        tagged_object_items = TaggedObject.objects.filter(
            object_id=object.id,
            content_type=ContentType.objects.get_for_model(django_model),
            tag__type=tag_type,
        ).select_related('tag')
        return [x.tag.value for x in tagged_object_items]


class TagTypeGenericInteger(TagTypeBase):
    """
        Числовой тип тега с произвольным смыслом, который в него может заложить пользователь
        Поддерживает операции ==, !=, <, >, <=, >=
        Это статический тип тега, то есть предполагается поиск его значения в таблице Tags
    """

    OPERATIONS = ["==", "!=", ">", "<", ">=", "<=", "range"]

    @staticmethod
    def is_dynamic():
        return False

    @staticmethod
    def get_semantic_type_name():
        return "GenericInteger"

    @staticmethod
    def get_semantic_type_name_l10n():
        return "Произвольное число"

    @staticmethod
    def get_language_type_name():
        return "integer"

    @staticmethod
    def get_db_type():
        return "GNRINT"

    @staticmethod
    def get_source_values_for_object(object):
        """
        Для переданного объекта ищем все теги с типом GenericInteger.
        :return: возвращаем массив, так как тегов такого типа может быть много
        """
        django_model = object._meta.model
        tag_type = TagTypeGenericInteger.get_db_type()
        if isinstance(object, User):
            return object.tag_values.get(tag_type, [])

        tagged_object_items = TaggedObject.objects.filter(
            object_id=object.id,
            content_type=ContentType.objects.get_for_model(django_model),
            tag__type=tag_type,
        ).select_related('tag')
        return [x.tag.value for x in tagged_object_items]

    @staticmethod
    def cast(obj):
        """
            Переопределяем метод базового класса для превращения строки в 'int',
            Иногда ( например в случае операции range ) в obj может быть строка вида:
                N1;N2 - диапазон чисел через ';'
            - в этом случае наш кастинг должен вернуть список дат из двух элементов
        """
        try:
            if re.match(r"^-?\d+;-?\d+$", str(obj)):
                nums = obj.split(';')
                return [int(x) for x in nums]
            else:
                return int(obj)
        except TypeError as e:
            # TODO: sentry
            raise RuntimeError("Casting object it not type of str: {}".format(str(e)))
        except ValueError as e:
            # TODO: sentry
            raise RuntimeError("String {} considered to have integer format".format(str(obj)))


class TagTypeCity(TagTypeBase):
    """
        Cтроковый тип тега, означающий название города.
        Поддерживает операции ==, !=
        Это статический тип тега, то есть предполагается поиск его значения в таблице Tags
    """

    OPERATIONS = ["==", "!="]

    @staticmethod
    def is_dynamic():
        return False

    @staticmethod
    def get_semantic_type_name():
        return "City"

    @staticmethod
    def get_semantic_type_name_l10n():
        return "Город"

    @staticmethod
    def get_language_type_name():
        return "string"

    @staticmethod
    def get_db_type():
        return "CTY"

    @staticmethod
    def get_source_values_for_object(object):
        """
            Для переданного объекта ищем все теги с типом City.
            :return: возвращаем массив, так как тегов такого типа может быть много
        """
        django_model = object._meta.model
        tag_type = TagTypeCity.get_db_type()
        if isinstance(object, User):
            return object.tag_values.get(tag_type, [])

        tagged_object_items = TaggedObject.objects.filter(
            object_id=object.id,
            content_type=ContentType.objects.get_for_model(django_model),
            tag__type=tag_type,
        ).select_related('tag')
        return [x.tag.value for x in tagged_object_items]


class TagTypeStudyGroup(TagTypeBase):
    """
        Cтроковый тип тега, означающий учебную группу.
        Поддерживает операции ==, !=
        Это статический тип тега, то есть предполагается поиск его значения в таблице Tags
    """

    OPERATIONS = ["==", "!="]

    @staticmethod
    def is_dynamic():
        return False

    @staticmethod
    def visible():
        return True

    @staticmethod
    def get_semantic_type_name():
        return "MoebiusGroup"

    @staticmethod
    def get_semantic_type_name_l10n():
        return "Учебная группа"

    @staticmethod
    def get_language_type_name():
        return "string"

    @staticmethod
    def get_db_type():
        return "MOEGRP"

    @staticmethod
    def get_source_values_for_object(object):
        """
            Для переданного объекта ищем все теги с типом StudyGroup.
            :return: возвращаем массив, так как тегов такого типа может быть много
        """
        django_model = object._meta.model
        tag_type = TagTypeStudyGroup.get_db_type()
        if isinstance(object, User):
            return object.tag_values.get(tag_type, [])

        tagged_object_items = TaggedObject.objects.filter(
            object_id=object.id,
            content_type=ContentType.objects.get_for_model(django_model),
            tag__type=tag_type,
        ).select_related('tag')
        return [x.tag.value for x in tagged_object_items]

    @classmethod
    def modify_suggest_query_param(cls, query_string):
        if len(query_string) != settings.STUDY_GROUP_SLUG_LEN:
            return None
        return query_string


class TagTypeUserNativeLang(TagTypeBase):
    """
        Cтроковый тип тега, означающий native-язык пользователя.
        Поддерживает операции ==, !=
        Значение нативного языка пользователя берем из staff_reader.get_user_language.
        Сильной нагрукзи на staff_reader не боимся, так как указанный метод обязуется кешировать результаты в redis.
    """

    OPERATIONS = ["==", "!="]

    @staticmethod
    def is_dynamic():
        return False

    @staticmethod
    def visible():
        return True

    @staticmethod
    def get_semantic_type_name():
        return "UserNativeLang"

    @staticmethod
    def get_semantic_type_name_l10n():
        return "Родной язык пользователя"

    @staticmethod
    def get_language_type_name():
        return "string"

    @staticmethod
    def get_db_type():
        return "USRNTLANG"

    @staticmethod
    def get_source_values_for_object(object):
        """
            Для переданного объекта ищем все теги с типом USRNTLANG.
            :return: возвращаем строку ('en', 'ru', ... )
        """
        return [staff_reader.get_user_native_language(username=object.username)]


class TagTypeAdapter(object):
    """
        Класс служит адаптером для использования в существующей django-модели.
        Предоставляет интерфейс получения различных представлений типов тегов,
        которые могут использоваться в моделях и вью django
    """
    __TAG_TYPES_AVAILABLE = [
        TagTypeGenericString,
        TagTypeGenericInteger,
        TagTypeCity,
        TagTypeStudyGroup,
        TagTypeProject,
        TagTypeCourse,
        TagTypeUser,
        TagTypeStaffGroup,
        TagTypeStaffStartDate,
        TagTypeStaffTimeFromStartDate,
        TagTypeIsChief,
        TagTypeChiefDuration,
        TagTypeUserNativeLang,
        TagTypeStaffCity,
        TagTypeStaffOffice,
        TagTypeStaffHRBP,
        TagTypeStaffChief,
    ]

    @staticmethod
    def get_available_tag_types():
        return [
            getattr(sys.modules[__name__], t.strip())
            for t in settings.TAG_TYPES_AVAILABLE.split(',')
        ]

    @staticmethod
    def get_available_dynamic_tag_types():
        return [x for x in TagTypeAdapter.get_available_tag_types() if x.is_dynamic() is True]

    @staticmethod
    def get_available_static_tag_types():
        return [x for x in TagTypeAdapter.get_available_tag_types() if x.is_dynamic() is False]

    @staticmethod
    def get_tag_db_types():
        return [x.get_db_type() for x in TagTypeAdapter.__TAG_TYPES_AVAILABLE]

    @staticmethod
    def get_tag_type_choices():
        return [
            [
                x.get_db_type(),
                x.get_semantic_type_name(),
            ]
            for x in TagTypeAdapter.__TAG_TYPES_AVAILABLE
        ]

    @staticmethod
    def get_language_tag_type_for_db_type(db_type):
        """
            Ищет среди доступных типов подходящий по строке DB-типа
            и возвращает соответствующий language-тип
        """
        for available_type in TagTypeAdapter.__TAG_TYPES_AVAILABLE:
            if available_type.get_db_type() == db_type:
                return available_type.get_language_type_name()

        # something strange if we haven't returned before
        return None

    @staticmethod
    def get_tag_type_for_db_type(db_type):
        """
            Ищет среди доступных типов подходящий по строке DB-типа
            и возвращает соответствующий тип
        """
        for available_type in TagTypeAdapter.__TAG_TYPES_AVAILABLE:
            if available_type.get_db_type() == db_type:
                return available_type

        # something strange if we haven't returned before
        return None

    @staticmethod
    def get_tag_type_for_semantic_type(semantic_type):
        """
            Ищет среди доступных типов подходящий по строке semantic-типа
            и возвращает соответствующий тип
        """
        for available_type in TagTypeAdapter.__TAG_TYPES_AVAILABLE:
            if available_type.get_semantic_type_name() == semantic_type:
                return available_type

        # something strange if we haven't returned before
        return None

    @staticmethod
    def get_dynamic_tag_types():
        return [x for x in TagTypeAdapter.__TAG_TYPES_AVAILABLE if x.is_dynamic() is True]


STAFF_ROLE_TAG_MAP = {
    'chief': TagTypeStaffChief,
    'hr_partner': TagTypeStaffHRBP,
}


@python_2_unicode_compatible
class Tag(models.Model):
    project = models.ForeignKey(
        'projects.Project',
        null=True,
        verbose_name=_('Проект'),
    )
    type = models.CharField(
        choices=TagTypeAdapter.get_tag_type_choices(),
        max_length=16,
        default=TagTypeGenericString.get_db_type(),
        verbose_name=_("Тип тега")
    )
    value = models.CharField(
        max_length=255,
        default="",
        verbose_name=_("Значение тега"),
    )
    data = JSONField(
        verbose_name=_("Дополнительные данные"),
        default={},
    )

    class Meta(object):
        verbose_name = _('Тег')
        verbose_name_plural = _('Теги')
        unique_together = ('project', 'value', 'type',)

    def __str__(self):
        return '{}: Type: {} Value: {}'.format(self.id, self.type, self.value)

    @property
    def semantic_type(self):
        """
            Возвращает семантическое представление типа тега
        """
        return TagTypeAdapter.get_tag_type_for_db_type(self.type).get_semantic_type_name()

    @property
    def semantic_type_l10n(self):
        """
            Возвращает локализованное семантическое представление типа тега
        """
        return TagTypeAdapter.get_tag_type_for_db_type(self.type).get_semantic_type_name_l10n()

    @property
    def internal_type(self):
        """
            Возвращает результат маппинга семантического типа в тип языка
        """
        return TagTypeAdapter.get_language_tag_type_for_db_type(self.type)


@python_2_unicode_compatible
class TaggedObject(models.Model):

    tag = models.ForeignKey(
        Tag,
        verbose_name=_("Тег"),
        on_delete=models.PROTECT,
        related_name="tagged_objects"
    )
    content_type = models.ForeignKey(
        ContentType,
        on_delete=models.PROTECT,
        related_name="tagged_objects",
        verbose_name=_("Тип тегированного объекта")
    )
    object_id = models.PositiveIntegerField(
        default=0,
        verbose_name=_("Идентификатор тегированного объекта")
    )
    content_object = GenericForeignKey(
        'content_type',
        'object_id'
    )
    start_time = models.DateTimeField(
        auto_now=False,
        auto_now_add=True,
        verbose_name=_("Дата начала действия"),
    )
    stop_time = models.DateTimeField(
        auto_now=False,
        auto_now_add=True,
        blank=True,
        null=True,
        verbose_name=_("Дата окончания действия"),
    )

    class Meta(object):
        verbose_name = _('Тегируемый объект')
        verbose_name_plural = _('Тегируемые объекты')

    def __str__(self):
        return '{}: Тип тега: {} Значение тега: {} Тип объекта: {} Id: {}'.format(
            self.id, self.tag.type, self.tag.value, self.content_type, self.object_id,
        )

    @staticmethod
    def _get_tags_for_queryset(queryset):
        """ Return sorted list of tags for given object_id and its content_type """
        tags = set()
        for tagged_object in queryset.select_related('tag'):
            tags.add(tagged_object.tag)
        return tags

    @staticmethod
    def get_tags_for_object(object_model, object_id):
        """ Return sorted list of tags for given object_id and its content_type """
        content_type = ContentType.objects.get_for_model(object_model)
        return TaggedObject._get_tags_for_queryset(
            TaggedObject.objects.filter(
                content_type=content_type,
                object_id=object_id)
        )

    @staticmethod
    def get_tags_for_type(object_model):
        """ Return sorted list of tags for given content_type """
        content_type = ContentType.objects.get_for_model(object_model)
        return TaggedObject._get_tags_for_queryset(
            TaggedObject.objects.filter(
                content_type=content_type,
            )
        )

    @staticmethod
    def get_matched_objects(object_id, object_model, match_model, select_related=None):
        """
            Returns QuerySet of matched object
            Input:
                object_id: id of object to be matched with objects of type 'match_model'
                object_model: model of object with id = object_id
                match_model: type of the objects to be matched with object_id
            Output:
                set of matched objects wich are of type 'match_model'
            Example:
                 get_matched_objects(123, User, Tag.COURSE) - this call should return all
                 courses which user with id=123 was assigned to
        """
        object_content_type = ContentType.objects.get_for_model(object_model)
        candidate_content_type = ContentType.objects.get_for_model(match_model)

        tagged_object_items = Tag.objects.filter(
            tagged_objects__object_id=object_id,
            tagged_objects__content_type=object_content_type,
        )

        # init tag groups
        tagged_object_groups = dict.fromkeys(TagTypeAdapter.get_tag_db_types())

        for tag_item in tagged_object_items:
            if tag_item.type in list(tagged_object_groups.keys()) and type(tagged_object_groups[tag_item.type]) == set:
                tagged_object_groups[tag_item.type].add(tag_item.value)
            else:
                tagged_object_groups[tag_item.type] = set([tag_item.value, ])

        candidates = match_model.objects.all()  # TODO: add get_for_project method for each model we want to match

        ids_to_include = []

        for candidate in candidates:

            candidate_tags = Tag.objects.filter(
                tagged_objects__object_id=candidate.id,
                tagged_objects__content_type=candidate_content_type,
            )
            if not candidate_tags:
                continue  # TODO: not tagged candidate
            candidates_objects_groups = dict.fromkeys(TagTypeAdapter.get_tag_db_types())
            for tag_item in candidate_tags:
                if (
                        tag_item.type in list(tagged_object_groups.keys()) and
                        type(candidates_objects_groups[tag_item.type]) == set
                ):
                    candidates_objects_groups[tag_item.type].add(tag_item.value)
                else:
                    candidates_objects_groups[tag_item.type] = set([tag_item.value, ])

            is_candidate_matched = True
            for k, v in candidates_objects_groups.items():
                if not v:
                    continue
                if not v & tagged_object_groups[k]:
                    # set intersection gives an empty set - so OR group not matched
                    is_candidate_matched = False
                    break

            if not is_candidate_matched:
                continue

            ids_to_include.append(candidate.id)

        return candidates.filter(id__in=ids_to_include)
