from builtins import range

from celery.utils.log import get_task_logger

from django.conf import settings
from django.core.paginator import Paginator

from kelvin.accounts.models import User
from kelvin.celery import app
from kelvin.common.decorators.task_logger import logged_task
from kelvin.projects.models import Project
from kelvin.tags.tasks_proc.groups import _tag_users_from_staff_groups_by_ids, _update_staff_group_tags
from kelvin.tags.tasks_proc.location import _tag_users_from_staff_location_by_ids, _update_staff_location_tags
from kelvin.tags.tasks_proc.role import _tag_users_from_staff_role_by_ids, _update_staff_role_tags

logger = get_task_logger(__name__)


@logged_task
@app.task()
def update_staff_location_tags(*args, **kwargs):
    _update_staff_location_tags(*args, **kwargs)


@logged_task
@app.task()
def force_renew_staff_location_tags():
    _update_staff_location_tags(force=True)


@logged_task
@app.task()
def _update_staff_role_tags_task(role: str):
    _update_staff_role_tags(role=role)


@logged_task
@app.task()
def update_staff_group_tags(*args, **kwargs):
    _update_staff_group_tags(*args, **kwargs)


@logged_task
@app.task()
def force_renew_staff_group_tags():
    _update_staff_group_tags(force=True)


@logged_task
@app.task()
def tag_users_from_staff_groups():
    """Асинхронная задача, создающая тэгирующая пользователей группами со стаффа.
    Задача слишком долгая, чтобы быть выполненной единственным воркером,
    поэтому она разбивается на множество маленьких при помощи пагинатора.
    """
    paginator = Paginator(
        User.objects.order_by('username').all().values_list('id', flat=True),
        settings.BATCH_COUNT_USERS_TO_UPDATE_FROM_STAFF,
    )
    for page_number in range(1, paginator.num_pages + 1):
        tag_users_from_staff_groups_by_ids.delay(
            piece_number=page_number,
            piece_count=paginator.num_pages,
            user_ids=list(paginator.page(page_number)),
            project_id=Project.objects.default().id,
        )


@logged_task
@app.task()
def tag_users_from_staff_groups_by_ids(user_ids, project_id=None, piece_number=None, piece_count=None):
    if project_id is None:
        project_id = Project.objects.default().id
    _tag_users_from_staff_groups_by_ids(user_ids, project_id)
    logger.info('User staff tags are updated ({} / {})'.format(piece_number, piece_count))


@logged_task
@app.task()
def tag_users_from_staff_location():
    """Асинхронная задача, создающая тэгирующая пользователей офисами и городами со стаффа.
    Задача слишком долгая, чтобы быть выполненной единственным воркером,
    поэтому она разбивается на множество маленьких при помощи пагинатора.
    """
    paginator = Paginator(
        User.objects.order_by('username').all().values_list('id', flat=True),
        settings.BATCH_COUNT_USERS_TO_UPDATE_FROM_STAFF,
    )
    for page_number in range(1, paginator.num_pages + 1):
        tag_users_from_staff_location_by_ids.delay(
            piece_number=page_number,
            piece_count=paginator.num_pages,
            user_ids=list(paginator.page(page_number)),
            project_id=Project.objects.default().id,
        )


@logged_task
@app.task()
def tag_users_from_staff_role(role):
    """Асинхронная задача, создающая тэгирующая пользователей hr-партнерами со стаффа.
    Задача слишком долгая, чтобы быть выполненной единственным воркером,
    поэтому она разбивается на множество маленьких при помощи пагинатора.
    """
    paginator = Paginator(
        User.objects.order_by('username').all().values_list('id', flat=True),
        settings.BATCH_COUNT_USERS_TO_UPDATE_FROM_STAFF,
    )
    for page_number in range(1, paginator.num_pages + 1):
        tag_users_from_staff_role_by_ids.delay(
            role=role,
            piece_number=page_number,
            piece_count=paginator.num_pages,
            user_ids=list(paginator.page(page_number)),
            project_id=Project.objects.default().id,
        )


@logged_task
@app.task()
def tag_users_from_staff_location_by_ids(user_ids, project_id=None, piece_number=None, piece_count=None):
    if project_id is None:
        project_id = Project.objects.default().id
    _tag_users_from_staff_location_by_ids(user_ids, project_id)
    logger.info('User staff location tags are updated ({} / {})'.format(piece_number, piece_count))


@logged_task
@app.task()
def tag_users_from_staff_role_by_ids(user_ids, role, project_id=None, piece_number=None, piece_count=None):
    if project_id is None:
        project_id = Project.objects.default().id
    _tag_users_from_staff_role_by_ids(user_ids, project_id, role)
    logger.info('User staff hrbp tags are updated ({} / {})'.format(piece_number, piece_count))
