from celery.utils.log import get_task_logger

from django.contrib.contenttypes.models import ContentType
from django.db import IntegrityError

from kelvin.accounts.models import User
from kelvin.common.staff_reader import staff_reader
from kelvin.projects.models import Project
from kelvin.tags.models import Tag, TaggedObject, TagTypeStaffGroup

logger = get_task_logger(__name__)


def _update_staff_group_tags(force=False):
    created_tags_count = 0
    deleted_tags_count = 0
    updated_tags_count = 0
    passed_tags_count = 0
    tag_ids_to_delete = []

    default_project = Project.objects.default()
    for group in staff_reader.get_all_groups():
        if not group['is_deleted']:
            existing_tag = Tag.objects.filter(
                project=default_project,
                type=TagTypeStaffGroup.get_db_type(),
                data__id=int(group['id']),
            ).first()
            if not existing_tag:
                tag_value = get_tag_value(group)

                duplicated_tag = Tag.objects.filter(
                    project=default_project,
                    type=TagTypeStaffGroup.get_db_type(),
                    value=tag_value,
                ).first()

                need_create_tag = True

                if duplicated_tag:
                    logger.info(u'Duplicated tag has found - {}'.format(duplicated_tag.value))
                    parent_name = duplicated_tag.data.get('parent_name', '')
                    duplicated_tag.value += ' ({})'.format(parent_name) if parent_name else ''
                    duplicated_tag.save()

                    tag_value = get_tag_value(group, use_parent_name=True)
                    if tag_value == duplicated_tag.value:
                        need_create_tag = False
                        logger.warning(u'Ignoring identical tags values - {}'.format(tag_value))

                if need_create_tag:
                    tag_data = get_group_data(group)
                    Tag.objects.create(
                        project=default_project,
                        type=TagTypeStaffGroup.get_db_type(),
                        value=tag_value,
                        data=tag_data,
                    )
                    created_tags_count += 1
            elif force:
                logger.info("Performing force data update for existing tag: {}".format(existing_tag.id))
                tag_data = get_group_data(group)
                existing_tag.data = tag_data
                existing_tag.save()
            elif is_update_needed(existing_tag, group):
                try:
                    existing_tag.value = get_tag_value(group)
                    existing_tag.data['parent_name'] = get_parent_name(group)
                    existing_tag.save()
                except IntegrityError:
                    existing_tag.value = get_tag_value(group, use_parent_name=True)
                    existing_tag.data['parent_name'] = get_parent_name(group)
                    existing_tag.save()
                updated_tags_count += 1
            else:
                #  Tag is up to date
                passed_tags_count += 1

        else:
            tag_ids_to_delete.append(int(group['id']))
            if len(tag_ids_to_delete) >= 100:
                deleted, _ = Tag.objects.filter(
                    data__id__in=tag_ids_to_delete,
                    # type=TagTypeStaffGroup.get_db_type()
                ).delete()
                deleted_tags_count += deleted

                tag_ids_to_delete = []

    deleted, _ = Tag.objects.filter(
        data__id__in=tag_ids_to_delete,
        # type=TagTypeStaffGroup.get_db_type()
    ).delete()
    deleted_tags_count += deleted

    logger.info(u'{} group tags are created'.format(created_tags_count))
    logger.info(u'{} group tags are deleted'.format(deleted_tags_count))
    logger.info(u'{} group tags are updated'.format(updated_tags_count))
    logger.info(u'{} group tags are passed'.format(passed_tags_count))

    logger.info(u'update_staff_group_tags() done!')


def _tag_users_from_staff_groups_by_ids(user_ids, project_id):
    """Асинхронная подзадача, создающая тэгирующая пользователей группами со стаффа.
    :param piece_number: Номер кусочка родительской задачи
    :param piece_count: Количество кусочков в родительской задаче
    :param user_ids: список id-шников пользователей, которые должны быть обновлены
    :param project_id: id проекта, в котором создаются тэги
    """
    tagged_objects_to_create, tagged_objects_to_delete = [], []
    for user in User.objects.filter(id__in=user_ids):
        to_create, to_delete = tag_single_user(user, project_id)
        tagged_objects_to_create += to_create
        tagged_objects_to_delete += to_delete

        if len(tagged_objects_to_create) > 1000:
            TaggedObject.objects.bulk_create(tagged_objects_to_create)
            tagged_objects_to_create = []

        if len(tagged_objects_to_delete) > 1000:
            TaggedObject.objects.filter(id__in=[to.id for to in tagged_objects_to_delete]).delete()
            tagged_objects_to_delete = []

    TaggedObject.objects.bulk_create(tagged_objects_to_create)
    TaggedObject.objects.filter(id__in=[to.id for to in tagged_objects_to_delete]).delete()


def get_group_data(group):
    return {
        'id': int(group['id']),
        'parent_name': group.get(
            'parent',
            {'name': ''}
        )['name'],
        'level': int(group.get('level', 0)),
        'url': group.get('url', ''),
    }


def get_tag_value(group, use_parent_name=False):
    if use_parent_name and 'parent' in group:
        return u'{} ({})'.format(group['name'], group['parent']['name'])
    return group['name']


def get_parent_name(group):
    return group.get('parent', {'name': ''})['name']


def is_update_needed(existing_tag, group):
    return (
        existing_tag.value not in (get_tag_value(group), get_tag_value(group, use_parent_name=True)) or
        existing_tag.data.get('parent_name') != get_parent_name(group)
    )


def tag_single_user(user, project_id):
    tagged_objects_to_create = []

    existing_tagged_objects = set(TaggedObject.objects.filter(
        tag__type=TagTypeStaffGroup.get_db_type(),
        content_type=ContentType.objects.get(app_label='accounts', model='user'),
        object_id=user.id,
    ))

    groups = staff_reader.get_user_groups(user.username)
    for group in groups:
        tagged_objects_to_create += tag_user_using_group(user, group['group'], existing_tagged_objects, project_id)

    # к этому моменту в existing_tagged_objects остались только те теги, которые надо удалить
    tagged_object_to_delete = existing_tagged_objects

    return tagged_objects_to_create, tagged_object_to_delete


def tag_user_using_group(user, group, existing_tagged_object_set, project_id):
    """Тэгирует пользователя группой со стаффа и ее родительскими группами
    :param user: тэгируемый пользователь
    :param group: словарь, описвыающий стафф группу
    :param project_id: id проекта, в котором создаются тэги
    :param existing_tagged_object_set: множество уже существующих тэгированных объектов,
        использующееся для удаления лишних
    :return: список TaggedObject, ещё не созданных в базе
    """
    tagged_objects_to_create = []
    if (
        'type' in group and group['type'] == 'department' and
        'is_deleted' in group and not group['is_deleted']
    ):
        tag = Tag.objects.filter(
            data__id=group['id'],
            project_id=project_id,
            type=TagTypeStaffGroup.get_db_type(),
        ).first()

        if tag:

            tagged_objects = set(
                to for to in existing_tagged_object_set
                if to.tag == tag
            )

            if tagged_objects:
                existing_tagged_object_set -= tagged_objects
            else:
                tagged_objects_to_create.append(
                    TaggedObject(
                        tag=tag,
                        content_type=ContentType.objects.get(app_label='accounts', model='user'),
                        object_id=user.id,
                    )
                )

            if 'ancestors' in group:
                for ancestor in group['ancestors']:
                    tagged_objects_to_create += tag_user_using_group(
                        user, ancestor, existing_tagged_object_set, project_id,
                    )

        else:
            group_tag_value = get_tag_value(group)
            logger.error('Group {}: {} not found (user: {})'.format(group['id'], group_tag_value, user.id))

    return tagged_objects_to_create
