from celery.utils.log import get_task_logger

from django.contrib.contenttypes.models import ContentType

from kelvin.accounts.models import User
from kelvin.common.staff_reader import staff_reader
from kelvin.projects.models import Project
from kelvin.tags.models import Tag, TaggedObject, TagTypeStaffCity, TagTypeStaffOffice

logger = get_task_logger(__name__)


def _update_staff_location_tags(force=False):
    created_office_tags_count = 0
    deleted_office_tags_count = 0
    updated_office_tags_count = 0
    passed_office_tags_count = 0
    office_tag_ids_to_delete = []

    created_city_tags_count = 0
    deleted_city_tags_count = 0
    updated_city_tags_count = 0
    passed_city_tags_count = 0
    city_tag_ids_to_delete = []

    default_project = Project.objects.default()

    for office in staff_reader.get_all_offices():
        if not office['is_deleted']:
            office_tag_value = get_office_value(office)
            existing_office_tag = Tag.objects.filter(
                project=default_project,
                type=TagTypeStaffOffice.get_db_type(),
                value=office_tag_value,
            ).first()
            if not existing_office_tag:
                office_tag_data = get_office_data(office)
                Tag.objects.create(
                    project=default_project,
                    type=TagTypeStaffOffice.get_db_type(),
                    value=office_tag_value,
                    data=office_tag_data,
                )
                created_office_tags_count += 1
            elif force:
                logger.info("Performing force data update for existing office tag: {}".format(existing_office_tag.id))
                office_tag_data = get_office_data(office)
                existing_office_tag.data = office_tag_data
                existing_office_tag.save()
            else:
                #  Tag is up to date
                passed_office_tags_count += 1

        else:
            office_tag_ids_to_delete.append(office['id'])
            if len(office_tag_ids_to_delete) >= 100:
                deleted, _ = Tag.objects.filter(data__id__in=office_tag_ids_to_delete).delete()
                deleted_office_tags_count += deleted

                office_tag_ids_to_delete = []

        city = office['city']
        if city and not city.get('is_deleted', True):
            existing_city_tag = Tag.objects.filter(
                project=default_project,
                type=TagTypeStaffCity.get_db_type(),
                data__id=city['id'],
            ).first()
            if not existing_city_tag:
                city_tag_value = get_city_value(city)
                city_tag_data = get_city_data(city)
                Tag.objects.create(
                    project=default_project,
                    type=TagTypeStaffCity.get_db_type(),
                    value=city_tag_value,
                    data=city_tag_data,
                )
                created_city_tags_count += 1
            elif force:
                logger.info("Performing force data update for existing city tag: {}".format(existing_city_tag.id))
                city_tag_data = get_city_data(city)
                existing_city_tag.data = city_tag_data
                existing_city_tag.save()
            else:
                #  Tag is up to date
                passed_city_tags_count += 1

        else:
            city_tag_ids_to_delete.append(city['id'])
            if len(city_tag_ids_to_delete) >= 100:
                deleted, _ = Tag.objects.filter(data__id__in=city_tag_ids_to_delete).delete()
                deleted_city_tags_count += deleted

                city_tag_ids_to_delete = []

    deleted, _ = Tag.objects.filter(
        data__id__in=office_tag_ids_to_delete, type=TagTypeStaffOffice.get_db_type()
    ).delete()
    deleted_office_tags_count += deleted

    deleted, _ = Tag.objects.filter(
        data__id__in=city_tag_ids_to_delete, type=TagTypeStaffCity.get_db_type()
    ).delete()
    deleted_city_tags_count += deleted

    logger.info(u'{} office tags are created'.format(created_office_tags_count))
    logger.info(u'{} office tags are deleted'.format(deleted_office_tags_count))
    logger.info(u'{} office tags are updated'.format(updated_office_tags_count))
    logger.info(u'{} office tags are passed'.format(passed_office_tags_count))

    logger.info(u'{} city tags are created'.format(created_city_tags_count))
    logger.info(u'{} city tags are deleted'.format(deleted_city_tags_count))
    logger.info(u'{} city tags are updated'.format(updated_city_tags_count))
    logger.info(u'{} city tags are passed'.format(passed_city_tags_count))

    logger.info(u'update_staff_location_tags() done!')


def get_office_value(office):
    return office['name']['ru']


def get_city_value(city):
    return city['name']['ru']


def get_office_data(office):
    return office


def get_city_data(city):
    return city


def _tag_users_from_staff_location_by_ids(user_ids, project_id):
    """Асинхронная подзадача, создающая тэгирующая пользователей офисами и городами со стаффа.
    :param piece_number: Номер кусочка родительской задачи
    :param piece_count: Количество кусочков в родительской задаче
    :param user_ids: список id-шников пользователей, которые должны быть обновлены
    :param project_id: id проекта, в котором создаются тэги
    """
    tagged_objects_to_create, tagged_objects_to_delete = [], []
    for user in User.objects.filter(id__in=user_ids):
        to_create, to_delete = tag_single_user(user, project_id)
        tagged_objects_to_create += to_create
        tagged_objects_to_delete += to_delete

        if len(tagged_objects_to_create) > 1000:
            TaggedObject.objects.bulk_create(tagged_objects_to_create)
            tagged_objects_to_create = []

        if len(tagged_objects_to_delete) > 1000:
            TaggedObject.objects.filter(id__in=[to.id for to in tagged_objects_to_delete]).delete()
            tagged_objects_to_delete = []

    TaggedObject.objects.bulk_create(tagged_objects_to_create)
    TaggedObject.objects.filter(id__in=[to.id for to in tagged_objects_to_delete]).delete()


def tag_single_user(user, project_id):
    tagged_objects_office_to_create = []
    tagged_objects_city_to_create = []

    existing_tagged_objects_office = set(TaggedObject.objects.filter(
        tag__type=TagTypeStaffOffice.get_db_type(),
        content_type=ContentType.objects.get(app_label='accounts', model='user'),
        object_id=user.id,
    ))
    existing_tagged_objects_city = set(TaggedObject.objects.filter(
        tag__type=TagTypeStaffCity.get_db_type(),
        content_type=ContentType.objects.get(app_label='accounts', model='user'),
        object_id=user.id,
    ))

    location = staff_reader.get_user_location(user.username)
    if 'office' not in location:
        logger.warning(u'User %s not found in staff', user.username)
        return ([], set())

    tagged_objects_office_to_create += tag_user_using_office(
        user, location['office'], existing_tagged_objects_office, project_id
    )
    tagged_objects_city_to_create += tag_user_using_city(
        user, location['office']['city'], existing_tagged_objects_city, project_id
    )

    # к этому моменту в existing_tagged_objects остались только те теги, которые надо удалить
    tagged_objects_office_to_delete = existing_tagged_objects_office
    tagged_objects_city_to_delete = existing_tagged_objects_city

    return (
        tagged_objects_office_to_create + tagged_objects_city_to_create,
        tagged_objects_office_to_delete & tagged_objects_city_to_delete,
    )


def tag_user_using_office(user, office, existing_tagged_object_set, project_id):
    """Тэгирует пользователя офисом со стаффа
    :param user: тэгируемый пользователь
    :param office: словарь, описвыающий стафф офис
    :param project_id: id проекта, в котором создаются тэги
    :param existing_tagged_object_set: множество уже существующих тэгированных объектов,
        использующееся для удаления лишних
    :return: список TaggedObject, ещё не созданных в базе
    """
    tagged_objects_to_create = []

    tag = Tag.objects.filter(
        data__id=office['id'],
        project_id=project_id,
        type=TagTypeStaffOffice.get_db_type(),
    ).first()

    if tag:
        tagged_objects = set(
            to for to in existing_tagged_object_set
            if to.tag == tag
        )

        if tagged_objects:
            existing_tagged_object_set -= tagged_objects
        else:
            tagged_objects_to_create.append(
                TaggedObject(
                    tag=tag,
                    content_type=ContentType.objects.get(app_label='accounts', model='user'),
                    object_id=user.id,
                )
            )
    else:
        office_tag_value = get_office_value(office)
        logger.error('Office {}: {} not found (user: {})'.format(office['id'], office_tag_value, user.id))

    return tagged_objects_to_create


def tag_user_using_city(user, city, existing_tagged_object_set, project_id):
    """Тэгирует пользователя офисом со стаффа
    :param user: тэгируемый пользователь
    :param city: словарь, описвыающий стафф город
    :param project_id: id проекта, в котором создаются тэги
    :param existing_tagged_object_set: множество уже существующих тэгированных объектов,
        использующееся для удаления лишних
    :return: список TaggedObject, ещё не созданных в базе
    """
    tagged_objects_to_create = []

    tag = Tag.objects.filter(
        data__id=city['id'],
        project_id=project_id,
        type=TagTypeStaffCity.get_db_type(),
    ).first()

    if tag:
        tagged_objects = set(
            to for to in existing_tagged_object_set
            if to.tag == tag
        )

        if tagged_objects:
            existing_tagged_object_set -= tagged_objects
        else:
            tagged_objects_to_create.append(
                TaggedObject(
                    tag=tag,
                    content_type=ContentType.objects.get(app_label='accounts', model='user'),
                    object_id=user.id,
                )
            )
    else:
        city_tag_value = get_city_value(city)
        logger.error('City {}: {} not found (user: {})'.format(city['id'], city_tag_value, user.id))

    return tagged_objects_to_create
