from celery.utils.log import get_task_logger

from django.contrib.contenttypes.models import ContentType

from kelvin.accounts.models import User
from kelvin.common.staff_reader import staff_reader
from kelvin.projects.models import Project
from kelvin.tags.models import STAFF_ROLE_TAG_MAP, Tag, TaggedObject, TagTypeStaffCity, TagTypeStaffOffice

logger = get_task_logger(__name__)


def _update_staff_role_tags(role):
    tag_class = STAFF_ROLE_TAG_MAP[role]
    default_project = Project.objects.default()

    existing_role_map = {int(tag.value): tag.id for tag in Tag.objects.filter(type=tag_class.get_db_type()).only('id', 'value')}
    tags_to_create = []
    tags_to_delete_ids = []
    for user in User.objects.only('id', 'username'):
        if staff_reader.has_department_role(user.username, role=role):
            if user.id not in existing_role_map:
                tags_to_create.append(
                    Tag(
                        project=default_project,
                        type=tag_class.get_db_type(),
                        value=str(user.id),
                        data=staff_reader.get_suggestuser(user.username),
                    ),
                )
        else:
            if user.id in existing_role_map:
                tags_to_delete_ids.append(existing_role_map[user.id])

    Tag.objects.bulk_create(tags_to_create)
    Tag.objects.filter(id__in=tags_to_delete_ids).delete()

    logger.info(f'{len(tags_to_create), tags_to_create} {role} tags are created')
    logger.info(f'{len(tags_to_delete_ids), tags_to_delete_ids} {role} tags are deleted')

    logger.info('_update_staff_role_tags() done!')


def _tag_users_from_staff_role_by_ids(user_ids, project_id, role):
    """Асинхронная подзадача, создающая тэгирующая пользователей HR-партнерами со стаффа.
    :param piece_number: Номер кусочка родительской задачи
    :param piece_count: Количество кусочков в родительской задаче
    :param user_ids: список id-шников пользователей, которые должны быть обновлены
    :param project_id: id проекта, в котором создаются тэги
    """
    tagged_objects_to_create, tagged_objects_to_delete = [], []
    for user in User.objects.filter(id__in=user_ids):
        to_create, to_delete = tag_single_user(user, project_id, role)

        tagged_objects_to_create += to_create
        tagged_objects_to_delete += to_delete

        if len(tagged_objects_to_create) > 1000:
            TaggedObject.objects.bulk_create(tagged_objects_to_create)
            tagged_objects_to_create = []

        if len(tagged_objects_to_delete) > 1000:
            TaggedObject.objects.filter(id__in=[to.id for to in tagged_objects_to_delete]).delete()
            tagged_objects_to_delete = []

    TaggedObject.objects.bulk_create(tagged_objects_to_create)
    TaggedObject.objects.filter(id__in=[to.id for to in tagged_objects_to_delete]).delete()


def tag_single_user(user, project_id, role):
    tag_class = STAFF_ROLE_TAG_MAP[role]
    field_in_staff = f'{role}s'
    tagged_objects_to_create = []

    existing_tagged_objects = set(TaggedObject.objects.filter(
        tag__type=tag_class.get_db_type(),
        content_type=ContentType.objects.get(app_label='accounts', model='user'),
        object_id=user.id,
    ))

    user_data = staff_reader.get_suggestuser(user.username)
    if field_in_staff not in user_data:
        logger.warning(u'User %s not found in staff', user.username)
        return ([], set())

    if len(user_data[field_in_staff]) == 0:
        return ([], set())

    role_data = user_data[field_in_staff][0]

    tagged_objects_to_create += tag_user_using_role(
        user, role_data, existing_tagged_objects, project_id, role
    )

    # к этому моменту в existing_tagged_objects остались только те теги, которые надо удалить
    tagged_objects_to_delete = existing_tagged_objects

    return (tagged_objects_to_create, tagged_objects_to_delete)


def tag_user_using_role(user, role_data, existing_tagged_object_set, project_id, role):
    """Тэгирует пользователя hr-партнером со стаффа
    :param user: тэгируемый пользователь
    :param office: словарь, описвыающий hr-партнера
    :param project_id: id проекта, в котором создаются тэги
    :param existing_tagged_object_set: множество уже существующих тэгированных объектов,
        использующееся для удаления лишних
    :return: список TaggedObject, ещё не созданных в базе
    """
    tag_class = STAFF_ROLE_TAG_MAP[role]
    tagged_objects_to_create = []

    tag = Tag.objects.filter(
        data__id=role_data['id'],
        project_id=project_id,
        type=tag_class.get_db_type(),
    ).first()


    if tag:
        tagged_objects = set(
            to for to in existing_tagged_object_set
            if to.tag == tag
        )

        if tagged_objects:
            existing_tagged_object_set -= tagged_objects
        else:
            tagged_objects_to_create.append(
                TaggedObject(
                    tag=tag,
                    content_type=ContentType.objects.get(app_label='accounts', model='user'),
                    object_id=user.id,
                )
            )
    else:
        tag_value = role_data['id']
        logger.error(f"{role} {role_data['id']}: {tag_value} not found (user: {user.id})")

    return tagged_objects_to_create
