from django.conf import settings
from django.contrib.contenttypes.models import ContentType
from django.core.cache import caches
from django.db.models.expressions import RawSQL

from kelvin.accounts.models import User
from kelvin.tags.models import Tag, TaggedObject, TagTypeStaffGroup

cache = caches['user_staff_groups']


def get_user_staff_groups(user_id):
    """
    Возвращает staff-группы пользователя, упорядоченные по их уровню иерархии
    """
    cache_key = settings.STAFF_GROUPS_CACHE_KEY.format(user_id)
    cached_user_staff_groups = cache.get(cache_key)

    if cached_user_staff_groups:
        return cached_user_staff_groups

    user_staff_group_tag_ids = [t.tag.id for t in TaggedObject.objects.filter(
        object_id=user_id,
        content_type=ContentType.objects.get_for_model(User),
        tag__type=TagTypeStaffGroup.get_db_type(),
    ).select_related('tag')]

    filtered_tags = Tag.objects.filter(id__in=user_staff_group_tag_ids)

    user_staff_group_tags = [x['value'] for x in filtered_tags.order_by(
        RawSQL(
            "data->>%s", ("level",)
        )
    ).values('value')]

    cache.set(
        cache_key,
        user_staff_group_tags,
        timeout=settings.USER_STAFF_GROUP_CACHE_TTL
    )

    return user_staff_group_tags
