from django.db.models.functions import Length, Lower

from rest_framework import status, viewsets
from rest_framework.decorators import list_route
from rest_framework.exceptions import ValidationError
from rest_framework.response import Response

from kelvin.accounts.utils import get_user_projects
from kelvin.tags.models import Tag, TagTypeAdapter
from kelvin.tags.serializers import TagSerializer, TagTypeSerializer


class TagTypeViewSet(viewsets.ViewSet):
    def list(self, request):
        available_types = [x for x in TagTypeAdapter.get_available_tag_types() if x.visible() is True]
        return Response(
            TagTypeSerializer(
                available_types,
                many=True
            ).data
        )


class TagViewSet(viewsets.ReadOnlyModelViewSet):
    serializer_class = TagSerializer

    @staticmethod
    def __validate_suggest_params(request):
        semantic_tag_type = request.query_params.get('semantic_type', None)
        q = request.query_params.get('q', None)
        if not q:
            raise ValidationError("'q' parameter is mandatory")
        if not semantic_tag_type:
            raise ValidationError("'semantic_type' parameter is mandatory")
        if not TagTypeAdapter.get_tag_type_for_semantic_type(semantic_tag_type):
            raise ValidationError('Unsupported semantic_type: {}'.format(semantic_tag_type))

    def get_queryset(self):
        query_string = self.request.query_params.get('q', '')
        tag_type = TagTypeAdapter.get_tag_type_for_semantic_type(
            self.request.query_params.get('semantic_type', '')
        )
        tag_db_type = tag_type.get_db_type()
        query_string = tag_type.modify_suggest_query_param(query_string)

        if not query_string:
            return Tag.objects.none()

        return Tag.objects.filter(
            project__in=get_user_projects(self.request.user),
            type=tag_db_type,
            value__icontains=query_string,
        ).order_by(Length('value'), Lower('value'))

    def get_user_project_queryset(self):
        return Tag.objects.filter(
            project__in=get_user_projects(self.request.user.id)
        ).order_by("value")

    @list_route(methods=['get'])
    def suggest(self, request):
        TagViewSet.__validate_suggest_params(self.request)
        tags = self.get_queryset()
        page = self.paginate_queryset(tags)
        qs = page if page is not None else tags
        response_data = self.get_serializer(qs, many=True).data
        if page is not None:
            return self.get_paginated_response(response_data)
        else:
            return Response(response_data)

    def retrieve(self, request):
        return Response(status=status.HTTP_405_METHOD_NOT_ALLOWED)

    def list(self, request):
        return Response(status=status.HTTP_405_METHOD_NOT_ALLOWED)
