import json
import logging
from builtins import str

from django.contrib.auth import get_user_model
from django.core.cache import cache

from rest_framework import status
from rest_framework.response import Response
from rest_framework.views import APIView

from .serializers import TakeoutUserSerializer

logger = logging.getLogger(__name__)
User = get_user_model()


class TakeoutView(APIView):
    """
    View to takeout gdrp users data
    """

    http_method_names = ["post"]

    NO_DATA_RESPONSE = {"status": "no_data"}

    CACHE_TIME = 60 * 10  # 10 minutes
    CACHE_NAME_TEMPLATE = "takeout_uid_{}"

    def post(self, request):
        """
        API point for GDPR Takeout
        Must return correct response format (otherwise causes indefinite retry)
        """

        uid = request.data.get('uid', None)
        unixtime = request.data.get('unixtime', None)

        if not uid or not unixtime:
            return Response({
                "status": "error",
                "error": "Not all POST-params present"
            })

        logger.debug('Got request for uid %s, unixtime %s', uid, unixtime)

        cache_name = self.CACHE_NAME_TEMPLATE.format(uid)
        result = cache.get(cache_name)

        if result is not None:
            return Response(result)

        try:
            user = User.objects.filter(yauid=uid).first()
        except User.DoesNotExist:
            logger.debug('User not found by uid: {}'.format(str(uid)))
            return Response(
                data={
                    "status": "no_data"
                },
                status=status.HTTP_404_NOT_FOUND
            )

        try:
            key = "uid_{}.json".format(uid)

            response = {
                "data": {
                    key: json.dumps(
                        TakeoutUserSerializer(
                            user,
                            context={"request": request}
                        ).data
                    ),
                },
                "status": "ok",
            }
            cache.set(cache_name, response, self.CACHE_TIME)
            return Response(response)

        except Exception as e:
            logger.exception(e)
            return Response({
                "status": "error",
                "error": "Sorry, data temporarily unavailable"
            })
