from rest_condition import And, Or

from rest_framework import viewsets
from rest_framework.decorators import detail_route
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response
from rest_framework.status import HTTP_400_BAD_REQUEST

from kelvin.accounts.permissions import ObjectForContentManager, ObjectForTeacher
from kelvin.results.models import CourseLessonResult
from kelvin.results.permissions import ObjectCourseForOwner, ObjectResultAuthor


class UserFilesPermissionViewSet(viewsets.GenericViewSet):
    """
    Реализует ручку, которая проверяет наличие у пользователя прав на загрузку
    файлов в попытку
    """

    queryset = (
        CourseLessonResult.objects.all()
        .select_related('summary')
        .select_related('summary__clesson__course__owner')
    )

    # Пользователь должен быть авторизован и он либо ученик, создавший эту
    # попытку, либо контент-менеджер, либо учитель, который создал курс
    permission_classes = (
        IsAuthenticated,
        Or(
            ObjectForContentManager,
            ObjectResultAuthor,
            And(
                ObjectForTeacher,
                ObjectCourseForOwner,
            ),
        ),
    )

    @detail_route()
    def check_upload(self, request, pk):
        """
        Проверяет, что у пользователя есть права загружать файлы в данную
        попытку
        """

        # Для того, чтобы вызвались пермишенны
        course_lesson_result = self.get_object()

        return Response({
            'user_id': request.user.id,
        })

    @detail_route()
    def check_meta(self, request, pk):
        """
        Проверяет, что у пользователя есть права на просмотр файлов.
        Параметры:
          public_keys - список ключей через запятую
          check_keys - проверять ли public_keys
        """

        check_keys = int(request.query_params.get('check_keys', 0))

        course_lesson_result = self.get_object()

        if check_keys:
            all_files_keys = [_file.get('public_key') for _file in course_lesson_result.get_all_files()]

            public_keys = request.query_params.get('public_keys', '')
            public_keys = public_keys.split(',') if public_keys else []

            if set(public_keys) - set(all_files_keys) and public_keys:
                return Response(status=HTTP_400_BAD_REQUEST)

            return Response({
                'user_id': request.user.id,
                'public_keys': public_keys or all_files_keys,
            })

        return Response({
            'user_id': request.user.id,
        })

    @detail_route()
    def check_delete(self, request, pk):
        """
        Проверяет права пользователя на удаление файлов
        """
        course_lesson_result = self.get_object()

        result_files = course_lesson_result.get_all_files()
        all_files_keys = [_file.get('public_key') for _file in result_files]

        public_keys = request.query_params.get('public_keys', '')
        public_keys = [key for key in public_keys.split(',') if key != '']

        if set(public_keys) - set(all_files_keys) and public_keys:
            return Response(status=HTTP_400_BAD_REQUEST)

        return Response({
            'user_id': request.user.id,
            'public_keys': public_keys or all_files_keys,
        })
