import sform
import waffle

from uuid import uuid4

from django.core import validators
from django.core.exceptions import ValidationError

from ok.api.approvements.validators import validate_issue_key
from ok.api.core.errors import AlreadyExists, SimpleValidationError
from ok.api.core.fetchers import query_params_data_getter
from ok.api.core.fields import (
    JsonField,
    QueueMultipleChoiceField,
    ScenarioMultipleChoiceField,
    GroupMultipleSuggestField,
)
from ok.api.core.forms import BaseForm, UserValidationFormMixin, EMPTY_LABEL
from ok.api.core.validators import clean_staff_groups, clean_staff_group_urls
from ok.approvements.choices import (
    APPROVEMENT_ROLES,
    APPROVEMENT_SORTING_TYPES,
    APPROVEMENT_STAGE_APPROVEMENT_SOURCES,
    APPROVEMENT_STATUSES,
    APPROVEMENT_TYPES,
    APPROVEMENT_STAGE_ACTIVE_STATUSES,
    POSSIBLE_DISAPPROVAL_REASONS,
)
from ok.approvements.helpers import filter_approvements_by_roles
from ok.approvements.models import Approvement, ApprovementStage
from ok.scenarios.choices import SCENARIO_STATUSES
from ok.scenarios.models import Scenario
from ok.tracker.models import Queue
from ok.tracker.queues import get_queue_by_object_id
from ok.utils.strings import str_to_md5


class ApprovementScenarioHelperForm(BaseForm):

    scenario = sform.SuggestField(
        queryset=Scenario.objects.all(),
        label_fields=['slug'],
    )


class ApprovementStandaloneHelperForm(BaseForm):

    type = sform.ChoiceField(choices=APPROVEMENT_TYPES, default=APPROVEMENT_TYPES.tracker)
    object_id = sform.CharField(max_length=32)

    def clean(self):
        cleaned_data = super().clean()
        cleaned_data['create_comment'] = True
        if cleaned_data.get('type') == APPROVEMENT_TYPES.tracker:
            cleaned_data['tracker_queue'] = get_queue_by_object_id(cleaned_data.get('object_id'))
        return cleaned_data


class ApprovementFormBase(BaseForm):

    object_id = sform.CharField(max_length=32)
    uid = sform.CharField(max_length=32)
    groups = GroupMultipleSuggestField()

    FIELDS_STATE = {
        'object_id': sform.REQUIRED,
    }

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        if not waffle.switch_is_active('enable_groups_multiplesuggest_for_approvements'):
            self.fields['groups'] = sform.GridField(
                field_instance=sform.CharField(
                    max_length=255,
                    state=sform.REQUIRED,
                ),
            )

    def clean_uid(self, uid):
        return str_to_md5(uid)

    def clean_groups(self, groups):
        if not waffle.switch_is_active('enable_approvement_groups_validation'):
            return groups
        if not waffle.switch_is_active('enable_groups_multiplesuggest_for_approvements'):
            return clean_staff_group_urls(groups)
        return clean_staff_groups(groups)


class ApprovementChildStageForm(BaseForm):

    approver = sform.CharField(max_length=32)


class ApprovementParentStageForm(BaseForm):

    approver = sform.CharField(max_length=32)
    need_all = sform.BooleanField()
    is_with_deputies = sform.BooleanField()
    stages = sform.GridField(
        field_instance=sform.FieldsetField(ApprovementChildStageForm),
    )
    need_approvals = sform.IntegerField()

    def clean(self):
        cleaned_data = super().clean()

        need_all = cleaned_data.pop('need_all', None)
        is_with_deputies = cleaned_data.get('is_with_deputies')

        if is_with_deputies:
            if need_all or cleaned_data['need_approvals']:
                raise ValidationError(
                    message='Specify only 1 of fields: is_with_deputies, need_all, need_approvals',
                    code='redundant_params',
                )
            cleaned_data['need_approvals'] = 1
        elif need_all:
            cleaned_data['need_approvals'] = len(cleaned_data.get('stages', []))
        elif cleaned_data['need_approvals'] is None:
            cleaned_data['need_approvals'] = 1

        return cleaned_data


class ApprovementEditableParamsForm(UserValidationFormMixin, BaseForm):
    text = sform.CharField()
    stages = sform.GridField(
        field_instance=sform.FieldsetField(ApprovementParentStageForm),
    )
    is_parallel = sform.BooleanField()
    is_reject_allowed = sform.BooleanField()
    callback_url = sform.CharField()
    tvm_id = sform.IntegerField()
    disapproval_reasons = sform.GridField(
        field_instance=sform.CharField(max_length=255),
    )
    info_mails_to = sform.GridField(
        field_instance=sform.EmailField(),
    )

    def clean(self):
        cleaned_data = super().clean()

        disapproval_reasons = cleaned_data.get('disapproval_reasons')
        if disapproval_reasons:
            validate_disapproval_reasons(disapproval_reasons)
            cleaned_data['disapproval_reasons'] = list(set(disapproval_reasons))

        info_mails_receivers = cleaned_data.get('info_mails_to', [])
        for receiver_email in info_mails_receivers:
            if not receiver_email.endswith('@yandex-team.ru'):
                raise ValidationError(
                    '',
                    code='Wrong email domain. Must be @yandex-team.ru',
                )
        return cleaned_data


class ApprovementEditForm(ApprovementEditableParamsForm):

    def clean(self):
        raw_cleaned_data = super().clean()

        cleaned_data = {
            key: value
            for key, value in raw_cleaned_data.items()
            if value not in validators.EMPTY_VALUES or key == 'stages'
        }

        if not cleaned_data:
            raise sform.ValidationError('No known field chosen')

        return cleaned_data


class ApprovementCreateValidateForm(ApprovementFormBase, ApprovementEditableParamsForm,
                                    ApprovementScenarioHelperForm):

    type = sform.ChoiceField(choices=APPROVEMENT_TYPES, default=APPROVEMENT_TYPES.tracker)
    text = sform.CharField()
    users = sform.GridField(
        field_instance=sform.CharField(
            max_length=32,
            state=sform.REQUIRED,
        ),
    )
    is_auto_approving = sform.BooleanField(default=False)
    is_reject_allowed = sform.BooleanField(default=True)
    flow_name = sform.CharField()
    flow_context = JsonField()
    create_comment = sform.BooleanField(default=False)

    # Автора могут задавать только пользователи с определёнными правами,
    # сейчас эти права регулируются waffle-флагом
    author = sform.CharField()

    FIELDS_STATE = {}
    USER_FIELDS = (
        'users',
        'author',
    )

    def clean_uid(self, uid):
        if not uid:
            uid = uuid4().hex
        return super().clean_uid(uid)

    def _collect_logins(self):
        logins = super()._collect_logins()
        logins['stages'] = []

        for stage in self.cleaned_data.get('stages', []):
            if stage['approver']:
                logins['stages'].append(stage['approver'])
            for child in stage.get('stages', []):
                logins['stages'].append(child['approver'])
        return logins

    def get_field_state(self, name):
        if name == 'stages' and (self.data.get('users') or self.data.get('flow_name')):
            return sform.NORMAL
        return super().get_field_state(name)

    @classmethod
    def get_duplicate_info(cls, object_id, only_active=False):
        qs = Approvement.objects.filter(object_id=object_id)
        if only_active:
            qs = qs.exclude(status=APPROVEMENT_STATUSES.closed)
        duplicate_info = qs.values('uuid', 'uid').first()
        return duplicate_info

    def clean_object_id(self, object_id):
        duplicate_info = self.get_duplicate_info(object_id, only_active=True)
        if duplicate_info:
            raise AlreadyExists(params=duplicate_info)
        return object_id

    def validate_tracker_fields(self):
        object_id = self.cleaned_data.get('object_id')
        if object_id:
            validate_issue_key(object_id)

    def clean(self):
        cleaned_data = super().clean()
        is_tracker_approvement = cleaned_data.get('type') == APPROVEMENT_TYPES.tracker
        if is_tracker_approvement:
            self.validate_tracker_fields()
        self.validate_user_fields(check_tracker=is_tracker_approvement)

        # Для обратной совместимости со старым фронтом переносим данные из users в stages
        # После OK-266 этот код можно будет удалить (+ удалить users из формы)
        if cleaned_data.get('users') and not cleaned_data.get('stages'):
            cleaned_data['stages'] = [{'approver': u} for u in cleaned_data['users']]
        cleaned_data.pop('users', None)

        return cleaned_data


class ApprovementCreateForm(ApprovementCreateValidateForm):

    FIELDS_STATE = {
        'object_id': sform.REQUIRED,
        'stages': sform.REQUIRED,
        'text': sform.REQUIRED,
    }


class ApprovementCreateFrontendForm(ApprovementCreateForm):

    # Note: при загрузке формы на фронте показываем допустимые сценарии, исходя из очереди.
    # Если предзаполнен сценарий, то показываем и его.
    scenario = sform.ModelChoiceField(
        queryset=Scenario.objects.none(),
        label_extractor=lambda x: x.name,
        empty_label=EMPTY_LABEL,
    )

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        scenario = self.initial.get('scenario')
        tracker_queue = self.initial.get('tracker_queue')
        if tracker_queue:
            scenario_slugs = (
                tracker_queue.scenario_macros.values('scenario_id')
                .union(tracker_queue.scenario_triggers.values('scenario_id'))
            )
            self.fields['scenario'].queryset = (
                Scenario.objects
                .filter(
                    slug__in=scenario_slugs,
                    status=SCENARIO_STATUSES.active,
                )
            )
        if scenario:
            self.fields['scenario'].queryset |= Scenario.objects.filter(slug=scenario)


class ApprovementListFilterBaseForm(BaseForm):

    default_getter = query_params_data_getter

    queues = QueueMultipleChoiceField()
    scenarios = ScenarioMultipleChoiceField()

    sort = sform.ChoiceField(
        choices=APPROVEMENT_SORTING_TYPES,
        default=APPROVEMENT_SORTING_TYPES.modified_asc,
        state=sform.REQUIRED,
    )

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        self.user = self.base_initial.get('user')
        assert self.user is not None

        user_approvements = self._get_user_approvements()

        self.fields['queues'].queryset = (
            Queue.objects
            .filter(id__in=user_approvements.values('tracker_queue'))
            .order_by('name')
        )
        self.fields['scenarios'].queryset = (
            Scenario.objects
            .filter(slug__in=user_approvements.filter(scenario__isnull=False).values('scenario'))
            .order_by('name')
        )

    def _get_user_approvements(self):
        raise NotImplementedError


class ApprovementListFilterForm(ApprovementListFilterBaseForm):

    stage_statuses = sform.MultipleChoiceField(
        choices=APPROVEMENT_STAGE_ACTIVE_STATUSES,
        default=[APPROVEMENT_STAGE_ACTIVE_STATUSES.current],
        state=sform.REQUIRED,
    )

    def _get_user_approvements(self):
        return Approvement.objects.filter(stages__approver=self.user.login)


class ApprovementForResponsiblesListFilterForm(ApprovementListFilterBaseForm):

    statuses = sform.MultipleChoiceField(
        choices=APPROVEMENT_STATUSES,
        default=[APPROVEMENT_STATUSES.in_progress],
        state=sform.REQUIRED,
    )

    def _get_user_approvements(self):
        return filter_approvements_by_roles(
            queryset=Approvement.objects,
            user=self.user,
            roles=[APPROVEMENT_ROLES.author],
        )


class ApprovementCheckForm(ApprovementFormBase):

    author = sform.CharField()

    FIELDS_STATE = ApprovementFormBase.FIELDS_STATE | {
        'uid': sform.REQUIRED,
    }

    def get_field_state(self, name):
        if name == 'author' and not self.data.get('groups'):
            return sform.REQUIRED
        return super().get_field_state(name)


class ApprovementApproveForm(BaseForm):

    # Авторизованный пользователь может не совпадать с согласующим данной стадии,
    # если он согласует не за себя
    approver = sform.CharField()

    stages = sform.MultipleSuggestField(
        queryset=ApprovementStage.objects.none(),
        label_fields=['approver'],
    )

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        assert 'active_stages' in self.initial
        self.fields['stages'].queryset = self.initial['active_stages']

    def get_field_state(self, name):
        if name == 'approver' and not self.data.get('stages'):
            return sform.REQUIRED
        return super().get_field_state(name)

    def clean(self):
        super().clean()
        self.cleaned_data['approvement_source'] = APPROVEMENT_STAGE_APPROVEMENT_SOURCES.api
        return self.cleaned_data


class ApprovementRejectForm(BaseForm):

    disapproval_reason = sform.CharField()

    def clean(self):
        super().clean()
        disapproval_reason = self.cleaned_data.get('disapproval_reason')
        if disapproval_reason:
            validate_disapproval_reasons([disapproval_reason])
        return self.cleaned_data


def validate_disapproval_reasons(disapproval_reasons):
    invalid_reasons = set(disapproval_reasons) - {slug for slug, name in POSSIBLE_DISAPPROVAL_REASONS}
    if invalid_reasons:
        raise ValidationError(
            '',
            code=f'Invalid disapproval reasons: {", ".join(str(reason) for reason in  invalid_reasons)}',
        )
