import waffle

from collections import OrderedDict

from django.utils.functional import cached_property
from django.utils.translation import ugettext_lazy as _
from rest_framework import serializers

from ok.api.core.serializers import SlugNameSerializer
from ok.approvements import models
from ok.approvements.stats import get_approvement_stage_stats, get_approvement_stats
from ok.approvements.workflow import ApprovementWorkflow
from ok.tracker.models import Queue


class QueueSerializer(serializers.ModelSerializer):

    class Meta:
        model = Queue
        fields = ('name', 'has_triggers', 'allow_externals')


class ApprovementCheckSerializer(serializers.Serializer):

    id = serializers.IntegerField()
    uuid = serializers.UUIDField()
    state = serializers.CharField()
    tracker_queue = QueueSerializer()


class ApprovementStageListItemSerializer(serializers.ModelSerializer):

    stages = serializers.SerializerMethodField()
    need_all = serializers.SerializerMethodField()

    class Meta:
        model = models.ApprovementStage
        fields = (
            'id',
            'approver',
            'is_approved',
            'approved_by',
            'approvement_source',
            'stages',
            'need_all',
            'need_approvals',
            'is_with_deputies',
            'position',
            'status',
        )

    def get_stages(self, obj):
        stages = getattr(obj, 'children', [])
        serializer = self.__class__(stages, many=True)
        serializer.bind('stages', self)
        return serializer.data

    def get_need_all(self, obj):
        need_approvals = getattr(obj, 'need_approvals', None)
        stages = getattr(obj, 'children', [])
        return need_approvals == len(stages) if need_approvals and stages else None


class ApprovementStageSerializer(ApprovementStageListItemSerializer):

    stats = serializers.SerializerMethodField()
    stages = serializers.SerializerMethodField()
    actions = serializers.SerializerMethodField()
    need_all = serializers.SerializerMethodField()

    class Meta:
        model = models.ApprovementStage
        fields = ApprovementStageListItemSerializer.Meta.fields + (
            'actions',
            'stats',
        )

    def get_stats(self, obj):
        return get_approvement_stage_stats(obj)

    def get_actions(self, obj):
        return {
            # Note: approve – псевдо-экшн/флаг, который указывает на то,
            # что данную стадию можно согласовывать от лица ответственного
            'approve': (
                obj.is_leaf
                and obj.is_active
                and self.root.cached_actions['approve']
                and self.root.cached_workflow.is_responsible
            ),
        }


class ApprovementListItemSerializer(serializers.ModelSerializer):

    stages = serializers.SerializerMethodField()
    actions = serializers.SerializerMethodField()
    text = serializers.SerializerMethodField()
    tracker_queue = QueueSerializer()
    scenario = SlugNameSerializer()

    stages_serializer_class = ApprovementStageListItemSerializer

    class Meta:
        model = models.Approvement
        fields = (
            'id',
            'uuid',
            'stages',
            'author',
            'groups',
            'text',
            'status',
            'resolution',
            'is_parallel',
            'url',
            'type',
            'object_id',
            'created',
            'modified',
            'tracker_queue',
            'is_auto_approving',
            'callback_url',
            'disapproval_reasons',
            'chosen_disapproval_reason',
            'info_mails_to',
            'actions',
            'scenario',
        )

    def get_stages(self, obj):
        # Note: Оптимизация запросов к базе: выбираем все стадии одним запросом
        # и сами делаем из них дерево. Перестанет работать как только нам понадобится
        # делать больше чем 2 уровня вложенных стадий.
        stages = OrderedDict()
        for stage in obj.stages.all():
            if stage.parent_id is None:
                stages[stage.id] = stage
                stages[stage.id].children = []
            else:
                stages[stage.parent_id].children.append(stage)

        serializer = self.stages_serializer_class(stages.values(), many=True)
        serializer.bind('stages', self)
        return serializer.data

    def get_actions(self, obj):
        # FIXME OK-902
        # Пока здесь сделаны псево-экшены, потому что в api списка согласований отдаются
        # только согласования, где ты текущий согласующий. В будущем это изменится,
        # а в OK-902 нужно придумать и сделать нормальный способ получения экшенов для списка
        return {
            'approve': True,
            'reject': obj.is_reject_allowed,
        }

    def get_text(self, obj):
        if not ('user' in self.root.context and 'request' in self.root.context):
            return obj.text

        if 'is_text_hiding_active' in self.root.context:
            is_text_hiding_active = self.root.context['is_text_hiding_active']
        else:
            is_text_hiding_active = waffle.flag_is_active(
                request=self.root.context['request'],
                flag_name='hide_approvement_text_for_observer',
            )
        if not is_text_hiding_active:
            return obj.text

        workflow = ApprovementWorkflow(obj, self.root.context['user'].login)
        if workflow.is_responsible or workflow.is_approver:
            return obj.text

        issue_url = 'https://st.yandex-team.ru/OK-1421'
        question = _('Why can\'t I see the text?')
        return f'** \*\*\*\*\*\*\*\*\*\* **\n(({issue_url} {question}))'


class ApprovementSerializer(ApprovementListItemSerializer):

    actions = serializers.SerializerMethodField()
    stats = serializers.SerializerMethodField()
    user_roles = serializers.SerializerMethodField()

    stages_serializer_class = ApprovementStageSerializer

    class Meta:
        model = models.Approvement
        fields = ApprovementListItemSerializer.Meta.fields + (
            'stats',
            'user_roles',
        )

    @cached_property
    def cached_workflow(self):
        if 'user' in self.root.context:
            return ApprovementWorkflow(self.instance, self.root.context['user'].login)
        return None

    @cached_property
    def cached_actions(self):
        return self.cached_workflow.get_actions() if self.cached_workflow else None

    def get_actions(self, obj):
        return self.cached_actions

    def get_stats(self, obj):
        return get_approvement_stats(obj)

    def get_user_roles(self, obj):
        return {
            'responsible': self.cached_workflow.is_responsible if self.cached_workflow else None,
            'approver': self.cached_workflow.is_active_approver if self.cached_workflow else None,
        }


class ApprovementApproveFormSerializer(serializers.Serializer):

    approver = serializers.SerializerMethodField()

    def get_approver(self, obj):
        user = self.root.context['user']
        if obj.author != user.login:
            return user.login
        approver = (
            obj.stages
            .active()
            .values_list('approver', flat=True)
            .first()
        )
        return approver or ''


class ApprovementRejectFormSerializer(serializers.Serializer):

    disapproval_reason = serializers.CharField(max_length=255, allow_null=True)
