import json

import waffle

from logging import getLogger

from constance import config
from startrek_client.exceptions import NotFound, StartrekError, Forbidden

from ok.api.core.errors import SimpleValidationError
from ok.tracker.base import get_user_tracker_client, client as tracker
from ok.tracker.queues import get_queue_name
from ok.utils.context import request_context


logger = getLogger(__name__)


def validate_issue_key(key):
    queue_name = get_queue_name(key)
    if not queue_name:
        logger.warning('Issue key is invalid: %s', key)
        raise SimpleValidationError('issue_does_not_exist')

    if waffle.switch_is_active('enable_issue_access_checking'):
        tracker_client = get_user_tracker_client()
    else:
        tracker_client = tracker

    try:
        tracker_client.issues.get(key=key, fields=['id'])
    except NotFound:
        logger.warning('Issue not found: %s', key)
        raise SimpleValidationError('issue_does_not_exist')
    except Forbidden:
        authors_whitelist = (
            json.loads(config.APPROVEMENT_AUTHORS_WHITELIST_BY_TRACKER_QUEUE)
            .get(queue_name, [])
        )
        if not request_context.ok_session_id and request_context.user in authors_whitelist:
            # логгируем пользователей, которым позволяем создать согласования без доступа к тикету
            logger.warning('User %s does not have access to issue %s', request_context.user, key)
        else:
            raise SimpleValidationError(code='permission_denied')
    except StartrekError as exc:
        logger.warning('Could not get issue %s: %s', key, exc)
        raise SimpleValidationError('startrek_error')
